/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Applies a demosaicing algorithm on grayscale image with color meaning acquired with a Bayer filter camera.
class IMAGEDEV_CPP_API BayerToRgb2d final : public GenericAlgorithm
{
public:
    /// The Bayer filter pattern.
    enum FilterMode
    {
    /// The Bayer pattern is GR/BG.
        GR_BG = 0,
    /// The Bayer pattern is GB/RG.
        GB_RG,
    /// The Bayer pattern is RG/GB.
        RG_GB,
    /// The Bayer pattern is BG/GR.
        BG_GR
    };
    /// The interpolation algorithm for demosaicing.
    enum InterpolationMode
    {
    /// Demosaicing is performed by bilinear interpolation.
        BILINEAR = 0,
    /// Demosaicing is performed by improved interpolation of color ratios.
        COLOR_RATIO,
    /// Demosaicing is performed by gradient and color ratios interpolation.
        GRADIENT
    };

    // Command constructor.
    BayerToRgb2d();


    /// Gets the inputGrayImage parameter.
    /// The Bayer grayscale input image.
    std::shared_ptr< iolink::ImageView > inputGrayImage() const;
    /// Sets the inputGrayImage parameter.
    /// The Bayer grayscale input image.
    void setInputGrayImage( std::shared_ptr< iolink::ImageView > inputGrayImage );

    /// Gets the filterMode parameter.
    /// The Bayer filter pattern.
    BayerToRgb2d::FilterMode filterMode() const;
    /// Sets the filterMode parameter.
    /// The Bayer filter pattern.
    void setFilterMode( const BayerToRgb2d::FilterMode& filterMode );

    /// Gets the interpolationMode parameter.
    /// The interpolation algorithm for demosaicing.
    BayerToRgb2d::InterpolationMode interpolationMode() const;
    /// Sets the interpolationMode parameter.
    /// The interpolation algorithm for demosaicing.
    void setInterpolationMode( const BayerToRgb2d::InterpolationMode& interpolationMode );

    /// Gets the outputColorImage parameter.
    /// The color output image. Its spatial dimensions and type are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputColorImage() const;
    /// Sets the outputColorImage parameter.
    /// The color output image. Its spatial dimensions and type are forced to the same values as the input.
    void setOutputColorImage( std::shared_ptr< iolink::ImageView > outputColorImage );

    // Method to launch the command.
    void execute();

};

/// Applies a demosaicing algorithm on grayscale image with color meaning acquired with a Bayer filter camera.
/// @param inputGrayImage The Bayer grayscale input image.
/// @param filterMode The Bayer filter pattern.
/// @param interpolationMode The interpolation algorithm for demosaicing.
/// @param outputColorImage The color output image. Its spatial dimensions and type are forced to the same values as the input.
/// @return Returns the outputColorImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
bayerToRgb2d( std::shared_ptr< iolink::ImageView > inputGrayImage,
              BayerToRgb2d::FilterMode filterMode,
              BayerToRgb2d::InterpolationMode interpolationMode,
              std::shared_ptr< iolink::ImageView > outputColorImage = nullptr );
} // namespace imagedev
