/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <ImageDev/Data/Cuda/CudaContext.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Smooths an image using an advanced edge preserving filter.
class IMAGEDEV_CPP_API CudaAdaptiveManifoldNlmFilter final : public GenericAlgorithm
{
public:

    // Command constructor.
    CudaAdaptiveManifoldNlmFilter();


    /// Gets the inputImage parameter.
    /// The input image. The image type can be integer or float.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image. The image type can be integer or float.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the spatialStandardDeviation parameter.
    /// The spatial standard deviation. Intuitively, it allows controlling how fast the similarity between two voxels decreases depending on their distance. The higher this value is, the blurrier the result is.
    double spatialStandardDeviation() const;
    /// Sets the spatialStandardDeviation parameter.
    /// The spatial standard deviation. Intuitively, it allows controlling how fast the similarity between two voxels decreases depending on their distance. The higher this value is, the blurrier the result is.
    void setSpatialStandardDeviation( const double& spatialStandardDeviation );

    /// Gets the intensityStandardDeviation parameter.
    /// The intensity/range standard deviation. Intuitively, it allows controlling how fast the similarity between two voxels decreases depending on the intensities of their neighborhoods.
    double intensityStandardDeviation() const;
    /// Sets the intensityStandardDeviation parameter.
    /// The intensity/range standard deviation. Intuitively, it allows controlling how fast the similarity between two voxels decreases depending on the intensities of their neighborhoods.
    void setIntensityStandardDeviation( const double& intensityStandardDeviation );

    /// Gets the kernelRadius parameter.
    /// The search window size to apply. The algorithm looks for matches within this area around each point.
    uint32_t kernelRadius() const;
    /// Sets the kernelRadius parameter.
    /// The search window size to apply. The algorithm looks for matches within this area around each point.
    void setKernelRadius( const uint32_t& kernelRadius );

    /// Gets the patchRadius parameter.
    /// Size of the neighborhood window. The influence of each point in the search window on the base point is weighted by comparing the neighborhood window of this point with the neighborhood window of the base point of the search window.
    uint32_t patchRadius() const;
    /// Sets the patchRadius parameter.
    /// Size of the neighborhood window. The influence of each point in the search window on the base point is weighted by comparing the neighborhood window of this point with the neighborhood window of the base point of the search window.
    void setPatchRadius( const uint32_t& patchRadius );

    /// Gets the cudaContext parameter.
    /// CUDA context information.
    CudaContext::Ptr cudaContext() const;
    /// Sets the cudaContext parameter.
    /// CUDA context information.
    void setCudaContext( const CudaContext::Ptr& cudaContext );

    /// Gets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Smooths an image using an advanced edge preserving filter.
/// @param inputImage The input image. The image type can be integer or float.
/// @param spatialStandardDeviation The spatial standard deviation. Intuitively, it allows controlling how fast the similarity between two voxels decreases depending on their distance. The higher this value is, the blurrier the result is.
/// @param intensityStandardDeviation The intensity/range standard deviation. Intuitively, it allows controlling how fast the similarity between two voxels decreases depending on the intensities of their neighborhoods.
/// @param kernelRadius The search window size to apply. The algorithm looks for matches within this area around each point.
/// @param patchRadius Size of the neighborhood window. The influence of each point in the search window on the base point is weighted by comparing the neighborhood window of this point with the neighborhood window of the base point of the search window.
/// @param cudaContext CUDA context information.
/// @param outputImage The output image. Its dimensions and type are forced to the same values as the input image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
cudaAdaptiveManifoldNlmFilter( std::shared_ptr< iolink::ImageView > inputImage,
                               double spatialStandardDeviation,
                               double intensityStandardDeviation,
                               uint32_t kernelRadius,
                               uint32_t patchRadius,
                               CudaContext::Ptr cudaContext,
                               std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
