/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <ImageDev/Data/Cuda/CudaContext.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a two-dimensional erosion using a structuring element matching with a square or a cross. The calculations are performed on the GPU.
/// Warning: This command is experimental, his signature may be modified between now and his final version.
class IMAGEDEV_CPP_API CudaErosion2d final : public GenericAlgorithm
{
public:
    /// The 2D neighborhood configuration.
    enum Neighborhood
    {
    /// The structuring element is a cross.
        CONNECTIVITY_4 = 0,
    /// The structuring element is a square.
        CONNECTIVITY_8
    };
    /// The way to manage the GPU memory.
    enum TilingMode
    {
    /// The entire input image is transferred to the GPU memory.
        NONE = 0,
    /// The input image is processed by tiles of a predefined size.
        USER_DEFINED
    };

    // Command constructor.
    CudaErosion2d();


    /// Gets the inputImage parameter.
    /// The input image. The image type can be integer or float.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image. The image type can be integer or float.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the kernelRadius parameter.
    /// The number of iterations (the half size of the structuring element, in pixels). A square structuring element always has an odd side length (3x3, 5x5, etc.) which is defined by twice the kernel radius + 1.
    uint32_t kernelRadius() const;
    /// Sets the kernelRadius parameter.
    /// The number of iterations (the half size of the structuring element, in pixels). A square structuring element always has an odd side length (3x3, 5x5, etc.) which is defined by twice the kernel radius + 1.
    void setKernelRadius( const uint32_t& kernelRadius );

    /// Gets the neighborhood parameter.
    /// The 2D neighborhood configuration.
    CudaErosion2d::Neighborhood neighborhood() const;
    /// Sets the neighborhood parameter.
    /// The 2D neighborhood configuration.
    void setNeighborhood( const CudaErosion2d::Neighborhood& neighborhood );

    /// Gets the tilingMode parameter.
    /// The way to manage the GPU memory.
    CudaErosion2d::TilingMode tilingMode() const;
    /// Sets the tilingMode parameter.
    /// The way to manage the GPU memory.
    void setTilingMode( const CudaErosion2d::TilingMode& tilingMode );

    /// Gets the tileSize parameter.
    /// The tile width and height in pixels. This parameter is used only in USER_DEFINED tiling mode.
    iolink::Vector2u32 tileSize() const;
    /// Sets the tileSize parameter.
    /// The tile width and height in pixels. This parameter is used only in USER_DEFINED tiling mode.
    void setTileSize( const iolink::Vector2u32& tileSize );

    /// Gets the cudaContext parameter.
    /// CUDA context information.
    CudaContext::Ptr cudaContext() const;
    /// Sets the cudaContext parameter.
    /// CUDA context information.
    void setCudaContext( const CudaContext::Ptr& cudaContext );

    /// Gets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Performs a two-dimensional erosion using a structuring element matching with a square or a cross. The calculations are performed on the GPU.
/// Warning: This command is experimental, his signature may be modified between now and his final version.
/// @param inputImage The input image. The image type can be integer or float.
/// @param kernelRadius The number of iterations (the half size of the structuring element, in pixels). A square structuring element always has an odd side length (3x3, 5x5, etc.) which is defined by twice the kernel radius + 1.
/// @param neighborhood The 2D neighborhood configuration.
/// @param tilingMode The way to manage the GPU memory.
/// @param tileSize The tile width and height in pixels. This parameter is used only in USER_DEFINED tiling mode.
/// @param cudaContext CUDA context information.
/// @param outputImage The output image. Its dimensions and type are forced to the same values as the input image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
cudaErosion2d( std::shared_ptr< iolink::ImageView > inputImage,
               uint32_t kernelRadius,
               CudaErosion2d::Neighborhood neighborhood,
               CudaErosion2d::TilingMode tilingMode,
               const iolink::Vector2u32& tileSize,
               CudaContext::Ptr cudaContext,
               std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
