/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/GlobalMeasurements/CylindricalIntensityProfileMsr.h>

namespace imagedev
{
/// Computes a radial intensity profile on a 3D image by performing a projection along an axis.
class IMAGEDEV_CPP_API CylindricalIntensityProfile3d final : public GenericAlgorithm
{
public:
    /// The axis along which slices are projected.
    enum AxisDirection
    {
    /// The YZ slices are projected.
        X_AXIS = 0,
    /// The XZ slices are projected.
        Y_AXIS,
    /// The XY slices are projected.
        Z_AXIS
    };
    /// The way to define the origin of the radii.
    enum CenterMode
    {
    /// The origin of the radii is the center of the input image.
        IMAGE_CENTER = 0,
    /// The origin of the radii is user defined.
        OTHER
    };

    // Command constructor.
    CylindricalIntensityProfile3d();


    /// Gets the inputImage parameter.
    /// The input image. Any type of image is accepted.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image. Any type of image is accepted.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the axisDirection parameter.
    /// The axis along which slices are projected.
    CylindricalIntensityProfile3d::AxisDirection axisDirection() const;
    /// Sets the axisDirection parameter.
    /// The axis along which slices are projected.
    void setAxisDirection( const CylindricalIntensityProfile3d::AxisDirection& axisDirection );

    /// Gets the centerMode parameter.
    /// The way to define the origin of the radii.
    CylindricalIntensityProfile3d::CenterMode centerMode() const;
    /// Sets the centerMode parameter.
    /// The way to define the origin of the radii.
    void setCenterMode( const CylindricalIntensityProfile3d::CenterMode& centerMode );

    /// Gets the center parameter.
    /// The origin of the radii in pixel coordinates. It is only used in OTHER mode.
    iolink::Vector3u32 center() const;
    /// Sets the center parameter.
    /// The origin of the radii in pixel coordinates. It is only used in OTHER mode.
    void setCenter( const iolink::Vector3u32& center );

    /// Gets the maximumRadius parameter.
    /// The radius in voxels up to which intensities are included in the profile.
    uint32_t maximumRadius() const;
    /// Sets the maximumRadius parameter.
    /// The radius in voxels up to which intensities are included in the profile.
    void setMaximumRadius( const uint32_t& maximumRadius );

    /// Gets the outputMeasurement parameter.
    /// The output object containing the computed radii and averaged intensities.
    CylindricalIntensityProfileMsr::Ptr outputMeasurement() const;

    // Method to launch the command.
    void execute();

};

/// Computes a radial intensity profile on a 3D image by performing a projection along an axis.
/// @param inputImage The input image. Any type of image is accepted.
/// @param axisDirection The axis along which slices are projected.
/// @param centerMode The way to define the origin of the radii.
/// @param center The origin of the radii in pixel coordinates. It is only used in OTHER mode.
/// @param maximumRadius The radius in voxels up to which intensities are included in the profile.
/// @param outputMeasurement The output object containing the computed radii and averaged intensities.
/// @return Returns the outputMeasurement output parameter.
IMAGEDEV_CPP_API 
CylindricalIntensityProfileMsr::Ptr
cylindricalIntensityProfile3d( std::shared_ptr< iolink::ImageView > inputImage,
                               CylindricalIntensityProfile3d::AxisDirection axisDirection,
                               CylindricalIntensityProfile3d::CenterMode centerMode,
                               const iolink::Vector3u32& center,
                               uint32_t maximumRadius,
                               CylindricalIntensityProfileMsr::Ptr outputMeasurement = nullptr );
} // namespace imagedev
