/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Computes a structure score image from the eigenvalues of a three-dimensional tensor image.
class IMAGEDEV_CPP_API EigenvaluesToStructureness3d final : public GenericAlgorithm
{
public:
    /// The type of structure lightness to extract.
    enum Lightness
    {
    /// Extracts bright structures from dark background.
        BRIGHT = 0,
    /// Extracts dark structures from bright background.
        DARK
    };
    /// The type of structure shapes to extract.
    enum StructureType
    {
    /// Extracts rod-like (tubular) structures.
        ROD = 0,
    /// Extracts blob-like (spherical) structures.
        BALL,
    /// Extracts plate-like (planar) structures.
        PLANE
    };

    // Command constructor.
    EigenvaluesToStructureness3d();


    /// Gets the inputEigenvaluesImage parameter.
    /// The input image containing the eigenvalues field. Type must be float. Spectral series size is 3 (channel 0 = largest eigenvalue, channel 1 = medium eigen value, channel 2 = smallest eigenvalue).
    std::shared_ptr< iolink::ImageView > inputEigenvaluesImage() const;
    /// Sets the inputEigenvaluesImage parameter.
    /// The input image containing the eigenvalues field. Type must be float. Spectral series size is 3 (channel 0 = largest eigenvalue, channel 1 = medium eigen value, channel 2 = smallest eigenvalue).
    void setInputEigenvaluesImage( std::shared_ptr< iolink::ImageView > inputEigenvaluesImage );

    /// Gets the lightness parameter.
    /// The type of structure lightness to extract.
    EigenvaluesToStructureness3d::Lightness lightness() const;
    /// Sets the lightness parameter.
    /// The type of structure lightness to extract.
    void setLightness( const EigenvaluesToStructureness3d::Lightness& lightness );

    /// Gets the structureType parameter.
    /// The type of structure shapes to extract.
    EigenvaluesToStructureness3d::StructureType structureType() const;
    /// Sets the structureType parameter.
    /// The type of structure shapes to extract.
    void setStructureType( const EigenvaluesToStructureness3d::StructureType& structureType );

    /// Gets the alpha parameter.
    /// The flatness sensitivity threshold. It corresponds to the alpha term of the score equation.
    double alpha() const;
    /// Sets the alpha parameter.
    /// The flatness sensitivity threshold. It corresponds to the alpha term of the score equation.
    void setAlpha( const double& alpha );

    /// Gets the beta parameter.
    /// The blobness sensitivity threshold. It corresponds to the beta term of the score equation.
    double beta() const;
    /// Sets the beta parameter.
    /// The blobness sensitivity threshold. It corresponds to the beta term of the score equation.
    void setBeta( const double& beta );

    /// Gets the noiseCutoff parameter.
    /// The noise scale factor. It is used for computing the c term of the score equation.
    double noiseCutoff() const;
    /// Sets the noiseCutoff parameter.
    /// The noise scale factor. It is used for computing the c term of the score equation.
    void setNoiseCutoff( const double& noiseCutoff );

    /// Gets the outputImage parameter.
    /// The score output image. X, Y, and Z dimensions, calibration and interpretation of the output image are forced to the same values as the input. The spectral dimension is forced to 1. Type is forced to float and all values are between 0 and 1.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The score output image. X, Y, and Z dimensions, calibration and interpretation of the output image are forced to the same values as the input. The spectral dimension is forced to 1. Type is forced to float and all values are between 0 and 1.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Computes a structure score image from the eigenvalues of a three-dimensional tensor image.
/// @param inputEigenvaluesImage The input image containing the eigenvalues field. Type must be float. Spectral series size is 3 (channel 0 = largest eigenvalue, channel 1 = medium eigen value, channel 2 = smallest eigenvalue).
/// @param lightness The type of structure lightness to extract.
/// @param structureType The type of structure shapes to extract.
/// @param alpha The flatness sensitivity threshold. It corresponds to the alpha term of the score equation.
/// @param beta The blobness sensitivity threshold. It corresponds to the beta term of the score equation.
/// @param noiseCutoff The noise scale factor. It is used for computing the c term of the score equation.
/// @param outputImage The score output image. X, Y, and Z dimensions, calibration and interpretation of the output image are forced to the same values as the input. The spectral dimension is forced to 1. Type is forced to float and all values are between 0 and 1.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
eigenvaluesToStructureness3d( std::shared_ptr< iolink::ImageView > inputEigenvaluesImage,
                              EigenvaluesToStructureness3d::Lightness lightness,
                              EigenvaluesToStructureness3d::StructureType structureType,
                              double alpha,
                              double beta,
                              double noiseCutoff,
                              std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
