/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the elasticRegistration2d function.
struct ElasticRegistration2dOutput
{
    /// The moving image transformed by the output displacement field. Its dimensions are forced to the same values as the fixed image, its type is forced to floating point precision (32-bit).
    std::shared_ptr< iolink::ImageView > outputImage;
    /// The displacement field that maps a pixel of the moving image onto the corresponding pixel in the fixed image. Each pixel of the displacement image contains two values: the first one represents the displacement in the X direction and one in the Y direction. Its dimensions are forced to the same values as the moving image, and its type is forced to floating point precision (32-bit).
    std::shared_ptr< iolink::ImageView > outputDisplacementField;
};

/// Computes the optimal elastic transformation to register a two-dimensional a moving image onto a fixed image.
class IMAGEDEV_CPP_API ElasticRegistration2d final : public GenericAlgorithm
{
public:
    /// The type of metric used to measure the similarity/dissimilarity between the fixed and the transformed image.
    enum MetricType
    {
    /// The metric used is the Mean Squared Difference, it has to be minimized.
        MEAN_SQUARE_DIFFERENCE = 0,
    /// The metric used is the Normalized Correlation, it has to be maximized.
        CORRELATION
    };

    // Command constructor.
    ElasticRegistration2d();


    /// Gets the inputFixedImage parameter.
    /// The input fixed grayscale image on which the moving image has to be registered.
    std::shared_ptr< iolink::ImageView > inputFixedImage() const;
    /// Sets the inputFixedImage parameter.
    /// The input fixed grayscale image on which the moving image has to be registered.
    void setInputFixedImage( std::shared_ptr< iolink::ImageView > inputFixedImage );

    /// Gets the inputMovingImage parameter.
    /// The input moving grayscale image to be registered. Its dimensions and type can be different than the moving image input.
    std::shared_ptr< iolink::ImageView > inputMovingImage() const;
    /// Sets the inputMovingImage parameter.
    /// The input moving grayscale image to be registered. Its dimensions and type can be different than the moving image input.
    void setInputMovingImage( std::shared_ptr< iolink::ImageView > inputMovingImage );

    /// Gets the metricType parameter.
    /// The type of metric used to measure the similarity/dissimilarity between the fixed and the transformed image.
    ElasticRegistration2d::MetricType metricType() const;
    /// Sets the metricType parameter.
    /// The type of metric used to measure the similarity/dissimilarity between the fixed and the transformed image.
    void setMetricType( const ElasticRegistration2d::MetricType& metricType );

    /// Gets the tolerance parameter.
    /// The maximum relative variation of the metric to stop the process. This variation is recomputed at each iteration.
    double tolerance() const;
    /// Sets the tolerance parameter.
    /// The maximum relative variation of the metric to stop the process. This variation is recomputed at each iteration.
    void setTolerance( const double& tolerance );

    /// Gets the pyramidLevels parameter.
    /// The two-dimensional vector representing the coarsest (resp. finest) resolution level in its first (resp. second) element.
    iolink::Vector2u32 pyramidLevels() const;
    /// Sets the pyramidLevels parameter.
    /// The two-dimensional vector representing the coarsest (resp. finest) resolution level in its first (resp. second) element.
    void setPyramidLevels( const iolink::Vector2u32& pyramidLevels );

    /// Gets the elasticStandardDeviation parameter.
    /// The standard deviation of the gaussian kernel used to smooth the displacement field at each iteration.
    iolink::Vector2d elasticStandardDeviation() const;
    /// Sets the elasticStandardDeviation parameter.
    /// The standard deviation of the gaussian kernel used to smooth the displacement field at each iteration.
    void setElasticStandardDeviation( const iolink::Vector2d& elasticStandardDeviation );

    /// Gets the fluidStandardDeviation parameter.
    /// The standard deviation of the gaussian kernel used to smooth the forces field at each iteration. This parameter value is critical to reach large displacements.
    iolink::Vector2d fluidStandardDeviation() const;
    /// Sets the fluidStandardDeviation parameter.
    /// The standard deviation of the gaussian kernel used to smooth the forces field at each iteration. This parameter value is critical to reach large displacements.
    void setFluidStandardDeviation( const iolink::Vector2d& fluidStandardDeviation );

    /// Gets the outputImage parameter.
    /// The moving image transformed by the output displacement field. Its dimensions are forced to the same values as the fixed image, its type is forced to floating point precision (32-bit).
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The moving image transformed by the output displacement field. Its dimensions are forced to the same values as the fixed image, its type is forced to floating point precision (32-bit).
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    /// Gets the outputDisplacementField parameter.
    /// The displacement field that maps a pixel of the moving image onto the corresponding pixel in the fixed image. Each pixel of the displacement image contains two values: the first one represents the displacement in the X direction and one in the Y direction. Its dimensions are forced to the same values as the moving image, and its type is forced to floating point precision (32-bit).
    std::shared_ptr< iolink::ImageView > outputDisplacementField() const;
    /// Sets the outputDisplacementField parameter.
    /// The displacement field that maps a pixel of the moving image onto the corresponding pixel in the fixed image. Each pixel of the displacement image contains two values: the first one represents the displacement in the X direction and one in the Y direction. Its dimensions are forced to the same values as the moving image, and its type is forced to floating point precision (32-bit).
    void setOutputDisplacementField( std::shared_ptr< iolink::ImageView > outputDisplacementField );

    // Method to launch the command.
    void execute();

};

/// Computes the optimal elastic transformation to register a two-dimensional a moving image onto a fixed image.
/// @param inputFixedImage The input fixed grayscale image on which the moving image has to be registered.
/// @param inputMovingImage The input moving grayscale image to be registered. Its dimensions and type can be different than the moving image input.
/// @param metricType The type of metric used to measure the similarity/dissimilarity between the fixed and the transformed image.
/// @param tolerance The maximum relative variation of the metric to stop the process. This variation is recomputed at each iteration.
/// @param pyramidLevels The two-dimensional vector representing the coarsest (resp. finest) resolution level in its first (resp. second) element.
/// @param elasticStandardDeviation The standard deviation of the gaussian kernel used to smooth the displacement field at each iteration.
/// @param fluidStandardDeviation The standard deviation of the gaussian kernel used to smooth the forces field at each iteration. This parameter value is critical to reach large displacements.
/// @param outputImage The moving image transformed by the output displacement field. Its dimensions are forced to the same values as the fixed image, its type is forced to floating point precision (32-bit).
/// @param outputDisplacementField The displacement field that maps a pixel of the moving image onto the corresponding pixel in the fixed image. Each pixel of the displacement image contains two values: the first one represents the displacement in the X direction and one in the Y direction. Its dimensions are forced to the same values as the moving image, and its type is forced to floating point precision (32-bit).
/// @return Returns a ElasticRegistration2dOutput structure containing the outputImage and outputDisplacementField output parameters.
IMAGEDEV_CPP_API 
ElasticRegistration2dOutput
elasticRegistration2d( std::shared_ptr< iolink::ImageView > inputFixedImage,
                       std::shared_ptr< iolink::ImageView > inputMovingImage,
                       ElasticRegistration2d::MetricType metricType,
                       double tolerance,
                       const iolink::Vector2u32& pyramidLevels,
                       const iolink::Vector2d& elasticStandardDeviation,
                       const iolink::Vector2d& fluidStandardDeviation,
                       std::shared_ptr< iolink::ImageView > outputImage = nullptr,
                       std::shared_ptr< iolink::ImageView > outputDisplacementField = nullptr );
} // namespace imagedev
