/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Provides different algorithms to perform an edge detection based on the first derivative of a two-dimensional image.
class IMAGEDEV_CPP_API GradientMagnitude3d final : public GenericAlgorithm
{
public:
    /// The gradient operator to apply.
    enum GradientOperator
    {
    /// The gradient is computed by convolution with the derivative of a Gaussian filter.
        GAUSSIAN = 0,
    /// The gradient is computed using the Sobel algorithm.
        SOBEL,
    /// The gradient is computed using the Prewitt algorithm.
        PREWITT
    };
    /// The gradient operator to apply.
    enum FilterMode
    {
    /// This mode uses a Finite Impulse Response algorithm on the X, Y and Z directions. This mode is faster for small standard deviation values. The kernel size value is four times the standard deviation and cannot be smaller than 5.
        FINITE = 0,
    /// This mode uses an Infinite Impulse Response algorithm on the X, Y and Z directions. The computation time is constant with this implementation. This mode is more precise for computing the Gaussian filter.
        RECURSIVE
    };
    /// The output image type to provide.
    enum OutputType
    {
    /// The output image has same type as the input image. In the case of integer images, this mode can lead to a loss of precision.
        SAME_AS_INPUT = 0,
    /// The output image type is forced to floating point.
        FLOAT_32_BIT
    };
    /// The output image to compute.
    enum GradientMode
    {
    /// This option computes the amplitude as the maximum between the X and Y components.
        AMPLITUDE_MAXIMUM = 0,
    /// This option computes the Euclidean norm of the vector.
        AMPLITUDE_EUCLIDEAN
    };

    // Command constructor.
    GradientMagnitude3d();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the gradientOperator parameter.
    /// The gradient operator to apply.
    GradientMagnitude3d::GradientOperator gradientOperator() const;
    /// Sets the gradientOperator parameter.
    /// The gradient operator to apply.
    void setGradientOperator( const GradientMagnitude3d::GradientOperator& gradientOperator );

    /// Gets the standardDeviation parameter.
    /// The standard deviation of the gaussian operator defines the gradient sharpness. Low values provide sharp gradient.
    iolink::Vector3d standardDeviation() const;
    /// Sets the standardDeviation parameter.
    /// The standard deviation of the gaussian operator defines the gradient sharpness. Low values provide sharp gradient.
    void setStandardDeviation( const iolink::Vector3d& standardDeviation );

    /// Gets the filterMode parameter.
    /// The gradient operator to apply.
    GradientMagnitude3d::FilterMode filterMode() const;
    /// Sets the filterMode parameter.
    /// The gradient operator to apply.
    void setFilterMode( const GradientMagnitude3d::FilterMode& filterMode );

    /// Gets the outputType parameter.
    /// The output image type to provide.
    GradientMagnitude3d::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The output image type to provide.
    void setOutputType( const GradientMagnitude3d::OutputType& outputType );

    /// Gets the gradientMode parameter.
    /// The output image to compute.
    GradientMagnitude3d::GradientMode gradientMode() const;
    /// Sets the gradientMode parameter.
    /// The output image to compute.
    void setGradientMode( const GradientMagnitude3d::GradientMode& gradientMode );

    /// Gets the outputImage parameter.
    /// The gradient amplitude output image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The gradient amplitude output image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Provides different algorithms to perform an edge detection based on the first derivative of a two-dimensional image.
/// @param inputImage The input image.
/// @param gradientOperator The gradient operator to apply.
/// @param standardDeviation The standard deviation of the gaussian operator defines the gradient sharpness. Low values provide sharp gradient.
/// @param filterMode The gradient operator to apply.
/// @param outputType The output image type to provide.
/// @param gradientMode The output image to compute.
/// @param outputImage The gradient amplitude output image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
gradientMagnitude3d( std::shared_ptr< iolink::ImageView > inputImage,
                     GradientMagnitude3d::GradientOperator gradientOperator,
                     const iolink::Vector3d& standardDeviation,
                     GradientMagnitude3d::FilterMode filterMode,
                     GradientMagnitude3d::OutputType outputType,
                     GradientMagnitude3d::GradientMode gradientMode,
                     std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
