/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Utilities/Deprecation.h>

namespace imagedev
{
// Output structure of the gradientOperator2d function.
struct GradientOperator2dOutput
{
    /// The X-gradient output image.
    std::shared_ptr< iolink::ImageView > outputImageX;
    /// The Y-gradient output image.
    std::shared_ptr< iolink::ImageView > outputImageY;
    /// The gradient amplitude output image.
    std::shared_ptr< iolink::ImageView > outputAmplitudeImage;
    /// The gradient orientation output image.
    std::shared_ptr< iolink::ImageView > outputOrientationImage;
};

/// Provides different algorithms to perform an edge detection based on the first derivative of a two-dimensional image.
class IMAGEDEV_CPP_API GradientOperator2d final : public GenericAlgorithm
{
public:
    /// The gradient operator to apply.
    enum GradientOperator
    {
    /// The gradient is computed using the Canny-Deriche algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM, AMPLITUDE_EUCLIDEAN, AMPLITUDE_AND_ORIENTATION and X_AND_Y_GRADIENTS.
        CANNY_DERICHE = 0,
    /// The gradient is computed using the Shen and Castan algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM.
        SHEN_CASTAN,
    /// The gradient is computed using the Sobel 3x3 kernel algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM, AMPLITUDE_EUCLIDEAN, AMPLITUDE_AND_ORIENTATION and X_AND_Y_GRADIENTS.
        KERNEL3X3,
    /// The gradient is computed using the Canny algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM and AMPLITUDE_EUCLIDEAN.
        CANNY,
    /// The gradient is computed using the Gaussian algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM, AMPLITUDE_EUCLIDEAN, AMPLITUDE_AND_ORIENTATION and X_AND_Y_GRADIENTS.
        GAUSSIAN,
    /// The gradient is computed using the Sobel algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM, AMPLITUDE_EUCLIDEAN, AMPLITUDE_AND_ORIENTATION and X_AND_Y_GRADIENTS.
        SOBEL,
    /// The gradient is computed using the Prewitt algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM, AMPLITUDE_EUCLIDEAN, AMPLITUDE_AND_ORIENTATION and X_AND_Y_GRADIENTS.
        PREWITT
    };
    /// The output image to compute.
    enum GradientMode
    {
    /// This option computes the gradient maximal amplitude. Only the outputAmplitudeImage output is set using this mode.
        AMPLITUDE_MAXIMUM = 0,
    /// This option computes the gradient Euclidean amplitude. Only the outputAmplitudeImage output is set using this mode.
        AMPLITUDE_EUCLIDEAN,
    /// This option computes Euclidean amplitude and orientation in degrees (between -128 and +128 degrees). Both outputAmplitudeImage and outputOrientationImage outputs are set using this mode.
        AMPLITUDE_AND_ORIENTATION,
    /// This option computes gradient in X and Y directions. Both outputImageX and outputImageY outputs are set using this mode.
        X_AND_Y_GRADIENTS
    };

    // Command constructor.
    IMAGEDEV_DEPRECATED( "This command is deprecated, it will be removed in ImageDev 2025.1. You can use GradientVector2d or GradientMagnitude2d instead.") GradientOperator2d();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the gradientOperator parameter.
    /// The gradient operator to apply.
    GradientOperator2d::GradientOperator gradientOperator() const;
    /// Sets the gradientOperator parameter.
    /// The gradient operator to apply.
    void setGradientOperator( const GradientOperator2d::GradientOperator& gradientOperator );

    /// Gets the gradientMode parameter.
    /// The output image to compute.
    GradientOperator2d::GradientMode gradientMode() const;
    /// Sets the gradientMode parameter.
    /// The output image to compute.
    void setGradientMode( const GradientOperator2d::GradientMode& gradientMode );

    /// Gets the smoothingFactor parameter.
    /// The smoothing factor defines the gradient sharpness. It is only used with Canny Deriche, Shen Castan, and Gaussian operators. It has a totally different meaning depending on the selected gradient operator. Its value must be between 0 and 100.
    double smoothingFactor() const;
    /// Sets the smoothingFactor parameter.
    /// The smoothing factor defines the gradient sharpness. It is only used with Canny Deriche, Shen Castan, and Gaussian operators. It has a totally different meaning depending on the selected gradient operator. Its value must be between 0 and 100.
    void setSmoothingFactor( const double& smoothingFactor );

    /// Gets the outputImageX parameter.
    /// The X-gradient output image.
    std::shared_ptr< iolink::ImageView > outputImageX() const;
    /// Sets the outputImageX parameter.
    /// The X-gradient output image.
    void setOutputImageX( std::shared_ptr< iolink::ImageView > outputImageX );

    /// Gets the outputImageY parameter.
    /// The Y-gradient output image.
    std::shared_ptr< iolink::ImageView > outputImageY() const;
    /// Sets the outputImageY parameter.
    /// The Y-gradient output image.
    void setOutputImageY( std::shared_ptr< iolink::ImageView > outputImageY );

    /// Gets the outputAmplitudeImage parameter.
    /// The gradient amplitude output image.
    std::shared_ptr< iolink::ImageView > outputAmplitudeImage() const;
    /// Sets the outputAmplitudeImage parameter.
    /// The gradient amplitude output image.
    void setOutputAmplitudeImage( std::shared_ptr< iolink::ImageView > outputAmplitudeImage );

    /// Gets the outputOrientationImage parameter.
    /// The gradient orientation output image.
    std::shared_ptr< iolink::ImageView > outputOrientationImage() const;
    /// Sets the outputOrientationImage parameter.
    /// The gradient orientation output image.
    void setOutputOrientationImage( std::shared_ptr< iolink::ImageView > outputOrientationImage );

    // Method to launch the command.
    void execute();

};

/// Provides different algorithms to perform an edge detection based on the first derivative of a two-dimensional image.
/// @param inputImage The input image.
/// @param gradientOperator The gradient operator to apply.
/// @param gradientMode The output image to compute.
/// @param smoothingFactor The smoothing factor defines the gradient sharpness. It is only used with Canny Deriche, Shen Castan, and Gaussian operators. It has a totally different meaning depending on the selected gradient operator. Its value must be between 0 and 100.
/// @param outputImageX The X-gradient output image.
/// @param outputImageY The Y-gradient output image.
/// @param outputAmplitudeImage The gradient amplitude output image.
/// @param outputOrientationImage The gradient orientation output image.
/// @return Returns a GradientOperator2dOutput structure containing the outputImageX, outputImageY, outputAmplitudeImage and outputOrientationImage output parameters.
IMAGEDEV_CPP_API 
GradientOperator2dOutput
gradientOperator2d( std::shared_ptr< iolink::ImageView > inputImage,
                    GradientOperator2d::GradientOperator gradientOperator,
                    GradientOperator2d::GradientMode gradientMode,
                    double smoothingFactor,
                    std::shared_ptr< iolink::ImageView > outputImageX = nullptr,
                    std::shared_ptr< iolink::ImageView > outputImageY = nullptr,
                    std::shared_ptr< iolink::ImageView > outputAmplitudeImage = nullptr,
                    std::shared_ptr< iolink::ImageView > outputOrientationImage = nullptr );
} // namespace imagedev
