/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the gradientVector2d function.
struct GradientVector2dOutput
{
    /// The gradient output image in X direction.
    std::shared_ptr< iolink::ImageView > outputImageX;
    /// The gradient output image in Y direction.
    std::shared_ptr< iolink::ImageView > outputImageY;
};

/// Provides different algorithms to perform an edge detection based on the first derivative of a two-dimensional image.
class IMAGEDEV_CPP_API GradientVector2d final : public GenericAlgorithm
{
public:
    /// The gradient operator to apply.
    enum GradientOperator
    {
    /// The gradient is computed by convolution with the derivative of a Gaussian filter.
        GAUSSIAN = 0,
    /// The gradient is computed using the Sobel algorithm.
        SOBEL,
    /// The gradient is computed using the Prewitt algorithm.
        PREWITT
    };
    /// The gradient operator to apply.
    enum FilterMode
    {
    /// This mode uses a Finite Impulse Response algorithm on the X and Y directions. This mode is faster for small standard deviation values. The kernel size value is four times the standard deviation and cannot be smaller than 5.
        FINITE = 0,
    /// This mode uses an Infinite Impulse Response algorithm on the X and Y directions. The computation time is constant with this implementation. This mode is more precise for computing the Gaussian filter.
        RECURSIVE
    };
    /// The output image type to provide.
    enum OutputType
    {
    /// The output image type is forced to 8-bit signed integer.
        SIGNED_INTEGER_8_BIT = 0,
    /// The output image type is forced to 16-bit signed integer.
        SIGNED_INTEGER_16_BIT,
    /// The output image type is forced to 32-bit signed integer.
        SIGNED_INTEGER_32_BIT,
    /// The output image type is forced to 32-bit float.
        FLOAT_32_BIT
    };

    // Command constructor.
    GradientVector2d();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the gradientOperator parameter.
    /// The gradient operator to apply.
    GradientVector2d::GradientOperator gradientOperator() const;
    /// Sets the gradientOperator parameter.
    /// The gradient operator to apply.
    void setGradientOperator( const GradientVector2d::GradientOperator& gradientOperator );

    /// Gets the standardDeviation parameter.
    /// The standard deviation of the gaussian operator defines the gradient sharpness. Low values provide sharp gradient.
    iolink::Vector2d standardDeviation() const;
    /// Sets the standardDeviation parameter.
    /// The standard deviation of the gaussian operator defines the gradient sharpness. Low values provide sharp gradient.
    void setStandardDeviation( const iolink::Vector2d& standardDeviation );

    /// Gets the filterMode parameter.
    /// The gradient operator to apply.
    GradientVector2d::FilterMode filterMode() const;
    /// Sets the filterMode parameter.
    /// The gradient operator to apply.
    void setFilterMode( const GradientVector2d::FilterMode& filterMode );

    /// Gets the outputType parameter.
    /// The output image type to provide.
    GradientVector2d::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The output image type to provide.
    void setOutputType( const GradientVector2d::OutputType& outputType );

    /// Gets the outputImageX parameter.
    /// The gradient output image in X direction.
    std::shared_ptr< iolink::ImageView > outputImageX() const;
    /// Sets the outputImageX parameter.
    /// The gradient output image in X direction.
    void setOutputImageX( std::shared_ptr< iolink::ImageView > outputImageX );

    /// Gets the outputImageY parameter.
    /// The gradient output image in Y direction.
    std::shared_ptr< iolink::ImageView > outputImageY() const;
    /// Sets the outputImageY parameter.
    /// The gradient output image in Y direction.
    void setOutputImageY( std::shared_ptr< iolink::ImageView > outputImageY );

    // Method to launch the command.
    void execute();

};

/// Provides different algorithms to perform an edge detection based on the first derivative of a two-dimensional image.
/// @param inputImage The input image.
/// @param gradientOperator The gradient operator to apply.
/// @param standardDeviation The standard deviation of the gaussian operator defines the gradient sharpness. Low values provide sharp gradient.
/// @param filterMode The gradient operator to apply.
/// @param outputType The output image type to provide.
/// @param outputImageX The gradient output image in X direction.
/// @param outputImageY The gradient output image in Y direction.
/// @return Returns a GradientVector2dOutput structure containing the outputImageX and outputImageY output parameters.
IMAGEDEV_CPP_API 
GradientVector2dOutput
gradientVector2d( std::shared_ptr< iolink::ImageView > inputImage,
                  GradientVector2d::GradientOperator gradientOperator,
                  const iolink::Vector2d& standardDeviation,
                  GradientVector2d::FilterMode filterMode,
                  GradientVector2d::OutputType outputType,
                  std::shared_ptr< iolink::ImageView > outputImageX = nullptr,
                  std::shared_ptr< iolink::ImageView > outputImageY = nullptr );
} // namespace imagedev
