/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a two-dimensional numerical reconstruction, starting from a grayscale marker image, into a grayscale input image.
class IMAGEDEV_CPP_API GrayscaleReconstruction2d final : public GenericAlgorithm
{
public:
    /// The type of reconstruction algorithm to apply.
    enum ReconstructionType
    {
    /// The reconstruction is performed by dilation. This mode removes the light points in a dark image.
        DILATION = 0,
    /// The reconstruction is performed by dilation. This mode removes the dark points in a light image.
        EROSION
    };
    /// The 2D neighborhood configuration for performing dilations or erosions.
    enum Neighborhood
    {
    /// The structuring element is a cross.
        CONNECTIVITY_4 = 0,
    /// The structuring element is a square.
        CONNECTIVITY_8
    };

    // Command constructor.
    GrayscaleReconstruction2d();


    /// Gets the inputImage parameter.
    /// The grayscale input mask image constraining reconstruction.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The grayscale input mask image constraining reconstruction.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputMarkerImage parameter.
    /// The grayscale input marker image containing seeds for reconstruction. It must have the same dimensions and type as the input image.
    std::shared_ptr< iolink::ImageView > inputMarkerImage() const;
    /// Sets the inputMarkerImage parameter.
    /// The grayscale input marker image containing seeds for reconstruction. It must have the same dimensions and type as the input image.
    void setInputMarkerImage( std::shared_ptr< iolink::ImageView > inputMarkerImage );

    /// Gets the reconstructionType parameter.
    /// The type of reconstruction algorithm to apply.
    GrayscaleReconstruction2d::ReconstructionType reconstructionType() const;
    /// Sets the reconstructionType parameter.
    /// The type of reconstruction algorithm to apply.
    void setReconstructionType( const GrayscaleReconstruction2d::ReconstructionType& reconstructionType );

    /// Gets the neighborhood parameter.
    /// The 2D neighborhood configuration for performing dilations or erosions.
    GrayscaleReconstruction2d::Neighborhood neighborhood() const;
    /// Sets the neighborhood parameter.
    /// The 2D neighborhood configuration for performing dilations or erosions.
    void setNeighborhood( const GrayscaleReconstruction2d::Neighborhood& neighborhood );

    /// Gets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Performs a two-dimensional numerical reconstruction, starting from a grayscale marker image, into a grayscale input image.
/// @param inputImage The grayscale input mask image constraining reconstruction.
/// @param inputMarkerImage The grayscale input marker image containing seeds for reconstruction. It must have the same dimensions and type as the input image.
/// @param reconstructionType The type of reconstruction algorithm to apply.
/// @param neighborhood The 2D neighborhood configuration for performing dilations or erosions.
/// @param outputImage The output image. Its dimensions and type are forced to the same values as the input.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
grayscaleReconstruction2d( std::shared_ptr< iolink::ImageView > inputImage,
                           std::shared_ptr< iolink::ImageView > inputMarkerImage,
                           GrayscaleReconstruction2d::ReconstructionType reconstructionType,
                           GrayscaleReconstruction2d::Neighborhood neighborhood,
                           std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
