/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a fast determination of the watershed lines in a three-dimensional grayscale image from a predefined set of markers.
class IMAGEDEV_CPP_API MarkerBasedWatershed3d final : public GenericAlgorithm
{
public:
    /// The mode for applying the watershed algorithm.
    enum AlgorithmMode
    {
    /// The result is repeatable but slower to compute.
        REPEATABLE = 0,
    /// The result is faster to compute but not repeatable because of asynchronous parallel computation. Since a watershed problem does not generally have a unique solution, two processings of the same image can lead to two different results (both exact).
        FAST
    };
    /// The type of output image.
    enum OutputType
    {
    /// The output is a binary image representing only the watershed lines.
        LINES = 0,
    /// The output is a label image representing only the marked basins. Voxels belonging to the watershed lines are filled with the most present label of their 26 neighbors.
        CONTIGUOUS_REGIONS,
    /// The output is a label image representing only the marked basins. Pixels belonging to the watershed lines are considered as background (set to 0).
        SEPARATED_REGIONS
    };
    /// The 3D neighborhood configuration defining the connected components.
    enum Neighborhood
    {
    /// Connected components are defined by a 6-neighbor connectivity. The connectivity pattern is composed of voxels sharing a common face with the voxel of interest.
        CONNECTIVITY_6 = 0,
    /// Connected components are defined by a 18-neighbor connectivity. The connectivity pattern is composed of voxels sharing a common edge with the voxel of interest.
        CONNECTIVITY_18,
    /// Connected components are defined by a 26-neighbor connectivity. The connectivity pattern is a full cube.
        CONNECTIVITY_26
    };

    // Command constructor.
    MarkerBasedWatershed3d();


    /// Gets the inputGrayImage parameter.
    /// The input grayscale image representing the landscape of the watershed.
    std::shared_ptr< iolink::ImageView > inputGrayImage() const;
    /// Sets the inputGrayImage parameter.
    /// The input grayscale image representing the landscape of the watershed.
    void setInputGrayImage( std::shared_ptr< iolink::ImageView > inputGrayImage );

    /// Gets the inputMarkerImage parameter.
    /// The input label image of markers.
    std::shared_ptr< iolink::ImageView > inputMarkerImage() const;
    /// Sets the inputMarkerImage parameter.
    /// The input label image of markers.
    void setInputMarkerImage( std::shared_ptr< iolink::ImageView > inputMarkerImage );

    /// Gets the algorithmMode parameter.
    /// The mode for applying the watershed algorithm.
    MarkerBasedWatershed3d::AlgorithmMode algorithmMode() const;
    /// Sets the algorithmMode parameter.
    /// The mode for applying the watershed algorithm.
    void setAlgorithmMode( const MarkerBasedWatershed3d::AlgorithmMode& algorithmMode );

    /// Gets the outputType parameter.
    /// The type of output image.
    MarkerBasedWatershed3d::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The type of output image.
    void setOutputType( const MarkerBasedWatershed3d::OutputType& outputType );

    /// Gets the neighborhood parameter.
    /// The 3D neighborhood configuration defining the connected components.
    MarkerBasedWatershed3d::Neighborhood neighborhood() const;
    /// Sets the neighborhood parameter.
    /// The 3D neighborhood configuration defining the connected components.
    void setNeighborhood( const MarkerBasedWatershed3d::Neighborhood& neighborhood );

    /// Gets the outputImage parameter.
    /// The output binary image. Its dimensions are forced to the same values as the input image. Its type depends on the selected outputType and potentially on the highest label of the inputMarkerImage.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output binary image. Its dimensions are forced to the same values as the input image. Its type depends on the selected outputType and potentially on the highest label of the inputMarkerImage.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Performs a fast determination of the watershed lines in a three-dimensional grayscale image from a predefined set of markers.
/// @param inputGrayImage The input grayscale image representing the landscape of the watershed.
/// @param inputMarkerImage The input label image of markers.
/// @param algorithmMode The mode for applying the watershed algorithm.
/// @param outputType The type of output image.
/// @param neighborhood The 3D neighborhood configuration defining the connected components.
/// @param outputImage The output binary image. Its dimensions are forced to the same values as the input image. Its type depends on the selected outputType and potentially on the highest label of the inputMarkerImage.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
markerBasedWatershed3d( std::shared_ptr< iolink::ImageView > inputGrayImage,
                        std::shared_ptr< iolink::ImageView > inputMarkerImage,
                        MarkerBasedWatershed3d::AlgorithmMode algorithmMode,
                        MarkerBasedWatershed3d::OutputType outputType,
                        MarkerBasedWatershed3d::Neighborhood neighborhood,
                        std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
