/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Approximates an image gradient by using morphological operations.
class IMAGEDEV_CPP_API MorphologicalGradient2d final : public GenericAlgorithm
{
public:
    /// The computation mode of morphological gradient.
    enum GradientMode
    {
    /// The engine emphasizes the internal edges.
        INTERNAL = 0,
    /// The engine emphasizes the external edges.
        EXTERNAL,
    /// The engine emphasizes the edges on objects borders.
        CENTERED
    };
    /// The precision for computation for morphological operations.
    enum Precision
    {
    /// The mophological operations are computed with a fast mode .
        FASTER = 0,
    /// The mophological operations are computed with a precise mode.
        PRECISE
    };

    // Command constructor.
    MorphologicalGradient2d();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the kernelRadius parameter.
    /// The radius in pixels of the disk structuring element used for morphological erosion and dilation. In centered mode, it corresponds to the thickness of the extracted edges and to their half thickness in other modes.
    uint32_t kernelRadius() const;
    /// Sets the kernelRadius parameter.
    /// The radius in pixels of the disk structuring element used for morphological erosion and dilation. In centered mode, it corresponds to the thickness of the extracted edges and to their half thickness in other modes.
    void setKernelRadius( const uint32_t& kernelRadius );

    /// Gets the gradientMode parameter.
    /// The computation mode of morphological gradient.
    MorphologicalGradient2d::GradientMode gradientMode() const;
    /// Sets the gradientMode parameter.
    /// The computation mode of morphological gradient.
    void setGradientMode( const MorphologicalGradient2d::GradientMode& gradientMode );

    /// Gets the precision parameter.
    /// The precision for computation for morphological operations.
    MorphologicalGradient2d::Precision precision() const;
    /// Sets the precision parameter.
    /// The precision for computation for morphological operations.
    void setPrecision( const MorphologicalGradient2d::Precision& precision );

    /// Gets the outputImage parameter.
    /// The output image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Approximates an image gradient by using morphological operations.
/// @param inputImage The input image.
/// @param kernelRadius The radius in pixels of the disk structuring element used for morphological erosion and dilation. In centered mode, it corresponds to the thickness of the extracted edges and to their half thickness in other modes.
/// @param gradientMode The computation mode of morphological gradient.
/// @param precision The precision for computation for morphological operations.
/// @param outputImage The output image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
morphologicalGradient2d( std::shared_ptr< iolink::ImageView > inputImage,
                         uint32_t kernelRadius,
                         MorphologicalGradient2d::GradientMode gradientMode,
                         MorphologicalGradient2d::Precision precision,
                         std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
