/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a three-dimensional opening using a structuring element matching with a sphere.
class IMAGEDEV_CPP_API OpeningBall3d final : public GenericAlgorithm
{
public:
    /// The precision of the computation method.
    enum Precision
    {
    /// The operation is computed with the fast mode, which approximates a circular structuring element by combining erosions using 6, 18 and 26 neighborhoods.
        FASTER = 0,
    /// The operation is computed with a precise mode (slower), which ensures a real spherical structuring element.
        PRECISE
    };
    /// The border policy to apply.
    enum BorderPolicy
    {
    /// The limited mode is faster to compute, but can produce the unexpected results for particles close to the image border.
        LIMITED = 0,
    /// The Extended mode is slower to compute, but produces the expected results for particles close to the image border.
        EXTENDED
    };

    // Command constructor.
    OpeningBall3d();


    /// Gets the inputImage parameter.
    /// The input image. The image type can be integer or float. Default is NULL.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image. The image type can be integer or float. Default is NULL.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the kernelRadius parameter.
    /// The length of the sphere radius in voxels.
    uint32_t kernelRadius() const;
    /// Sets the kernelRadius parameter.
    /// The length of the sphere radius in voxels.
    void setKernelRadius( const uint32_t& kernelRadius );

    /// Gets the precision parameter.
    /// The precision of the computation method.
    OpeningBall3d::Precision precision() const;
    /// Sets the precision parameter.
    /// The precision of the computation method.
    void setPrecision( const OpeningBall3d::Precision& precision );

    /// Gets the borderPolicy parameter.
    /// The border policy to apply.
    OpeningBall3d::BorderPolicy borderPolicy() const;
    /// Sets the borderPolicy parameter.
    /// The border policy to apply.
    void setBorderPolicy( const OpeningBall3d::BorderPolicy& borderPolicy );

    /// Gets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Performs a three-dimensional opening using a structuring element matching with a sphere.
/// @param inputImage The input image. The image type can be integer or float. Default is NULL.
/// @param kernelRadius The length of the sphere radius in voxels.
/// @param precision The precision of the computation method.
/// @param borderPolicy The border policy to apply.
/// @param outputImage The output image. Its dimensions and type are forced to the same values as the input image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
openingBall3d( std::shared_ptr< iolink::ImageView > inputImage,
               uint32_t kernelRadius,
               OpeningBall3d::Precision precision,
               OpeningBall3d::BorderPolicy borderPolicy,
               std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
