/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/GlobalMeasurements/PercolationThreshold3dMsr.h>

namespace imagedev
{
/// Computes the intensity level spread following an axis of propagation in a three-dimensional image.
class IMAGEDEV_CPP_API PercolationThreshold3d final : public GenericAlgorithm
{
public:
    /// The axis along which the propagation is performed.
    enum Axis
    {
    /// The propagation is performed along the X axis.
        X_AXIS = 0,
    /// The propagation is performed along the Y axis.
        Y_AXIS,
    /// The propagation is performed along the Z axis.
        Z_AXIS
    };
    /// The 3D neighborhood configuration.
    enum Neighborhood
    {
    /// The structuring element is composed of voxels with a common face with the voxel of interest.
        CONNECTIVITY_6 = 0,
    /// The structuring element is composed of voxels with at least one common edge.
        CONNECTIVITY_18,
    /// The structuring element is a full cube.
        CONNECTIVITY_26
    };

    // Command constructor.
    PercolationThreshold3d();


    /// Gets the inputImage parameter.
    /// The input grayscale or color image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input grayscale or color image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the maskAndSeedValues parameter.
    /// The first value is a mask value. Voxels with this intensity value are not infiltrated by the propagation. The second value is the seed value. Voxels with an intensity value lower than this seed value are not infiltrated by the propagation.
    iolink::Vector2i32 maskAndSeedValues() const;
    /// Sets the maskAndSeedValues parameter.
    /// The first value is a mask value. Voxels with this intensity value are not infiltrated by the propagation. The second value is the seed value. Voxels with an intensity value lower than this seed value are not infiltrated by the propagation.
    void setMaskAndSeedValues( const iolink::Vector2i32& maskAndSeedValues );

    /// Gets the searchRange parameter.
    /// The range inside which the percolation threshold is searched.
    iolink::Vector2i32 searchRange() const;
    /// Sets the searchRange parameter.
    /// The range inside which the percolation threshold is searched.
    void setSearchRange( const iolink::Vector2i32& searchRange );

    /// Gets the axis parameter.
    /// The axis along which the propagation is performed.
    PercolationThreshold3d::Axis axis() const;
    /// Sets the axis parameter.
    /// The axis along which the propagation is performed.
    void setAxis( const PercolationThreshold3d::Axis& axis );

    /// Gets the neighborhood parameter.
    /// The 3D neighborhood configuration.
    PercolationThreshold3d::Neighborhood neighborhood() const;
    /// Sets the neighborhood parameter.
    /// The 3D neighborhood configuration.
    void setNeighborhood( const PercolationThreshold3d::Neighborhood& neighborhood );

    /// Gets the outputMeasurement parameter.
    /// The output measurement result.
    PercolationThreshold3dMsr::Ptr outputMeasurement() const;

    // Method to launch the command.
    void execute();

};

/// Computes the intensity level spread following an axis of propagation in a three-dimensional image.
/// @param inputImage The input grayscale or color image.
/// @param maskAndSeedValues The first value is a mask value. Voxels with this intensity value are not infiltrated by the propagation. The second value is the seed value. Voxels with an intensity value lower than this seed value are not infiltrated by the propagation.
/// @param searchRange The range inside which the percolation threshold is searched.
/// @param axis The axis along which the propagation is performed.
/// @param neighborhood The 3D neighborhood configuration.
/// @param outputMeasurement The output measurement result.
/// @return Returns the outputMeasurement output parameter.
IMAGEDEV_CPP_API 
PercolationThreshold3dMsr::Ptr
percolationThreshold3d( std::shared_ptr< iolink::ImageView > inputImage,
                        const iolink::Vector2i32& maskAndSeedValues,
                        const iolink::Vector2i32& searchRange,
                        PercolationThreshold3d::Axis axis,
                        PercolationThreshold3d::Neighborhood neighborhood,
                        PercolationThreshold3dMsr::Ptr outputMeasurement = nullptr );
} // namespace imagedev
