/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Filters a two-dimensional image by applying a radial background subtraction on the Fourier transform.
class IMAGEDEV_CPP_API RadialFrequencyFilter2d final : public GenericAlgorithm
{
public:
    /// The way to compute the background subtraction in the Fourier domain.
    enum FilterMode
    {
    /// The background image B is simply subtracted from the modulus of the Fourier transform |I|.
        SUBTRACT = 0,
    /// The square of the background image B is subtracted from the square of the modulus of the Fourier transform |I|.
        WIENER
    };
    /// The way to estimate the background model of the Fourier transform module.
    enum BackgroundModel
    {
    /// The background is estimated as a radial average of the Fourier transform of the whole image.
        AVERAGE = 0,
    /// The background is estimated by a polynomial fit to the radial average of the Fourier transform of the whole image.
        POLYNOME
    };

    // Command constructor.
    RadialFrequencyFilter2d();


    /// Gets the inputImage parameter.
    /// The input image. Its data type can be integer or floating point.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image. Its data type can be integer or floating point.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the filterMode parameter.
    /// The way to compute the background subtraction in the Fourier domain.
    RadialFrequencyFilter2d::FilterMode filterMode() const;
    /// Sets the filterMode parameter.
    /// The way to compute the background subtraction in the Fourier domain.
    void setFilterMode( const RadialFrequencyFilter2d::FilterMode& filterMode );

    /// Gets the backgroundModel parameter.
    /// The way to estimate the background model of the Fourier transform module.
    RadialFrequencyFilter2d::BackgroundModel backgroundModel() const;
    /// Sets the backgroundModel parameter.
    /// The way to estimate the background model of the Fourier transform module.
    void setBackgroundModel( const RadialFrequencyFilter2d::BackgroundModel& backgroundModel );

    /// Gets the removeHighFrequencies parameter.
    /// Indicates whether high frequencies outside the cut-off frequency should be removed by a circular mask. This post processing applies a low pass filter and can be useful for removing noise.
    bool removeHighFrequencies() const;
    /// Sets the removeHighFrequencies parameter.
    /// Indicates whether high frequencies outside the cut-off frequency should be removed by a circular mask. This post processing applies a low pass filter and can be useful for removing noise.
    void setRemoveHighFrequencies( const bool& removeHighFrequencies );

    /// Gets the cutoffFrequency parameter.
    /// Fraction of the Maximum Frequency of the image. A value of 0.5 implies that the circular mask has a diameter equal to the half width of the FFT image. This parameter is ignored if removeHighFrequencies is set to false. It must be a strictly positive value, lower than or equal to 1.
    double cutoffFrequency() const;
    /// Sets the cutoffFrequency parameter.
    /// Fraction of the Maximum Frequency of the image. A value of 0.5 implies that the circular mask has a diameter equal to the half width of the FFT image. This parameter is ignored if removeHighFrequencies is set to false. It must be a strictly positive value, lower than or equal to 1.
    void setCutoffFrequency( const double& cutoffFrequency );

    /// Gets the transitionBand parameter.
    /// Fraction of the cut-off frequency to define the width of the transition band. This parameter is ignored if removeHighFrequencies is set to false. It must be a positive or null value, lower than or equal to 1.
    double transitionBand() const;
    /// Sets the transitionBand parameter.
    /// Fraction of the cut-off frequency to define the width of the transition band. This parameter is ignored if removeHighFrequencies is set to false. It must be a positive or null value, lower than or equal to 1.
    void setTransitionBand( const double& transitionBand );

    /// Gets the outputImage parameter.
    /// The output image. The output image characteristics are forced to the same as the input image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. The output image characteristics are forced to the same as the input image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Filters a two-dimensional image by applying a radial background subtraction on the Fourier transform.
/// @param inputImage The input image. Its data type can be integer or floating point.
/// @param filterMode The way to compute the background subtraction in the Fourier domain.
/// @param backgroundModel The way to estimate the background model of the Fourier transform module.
/// @param removeHighFrequencies Indicates whether high frequencies outside the cut-off frequency should be removed by a circular mask. This post processing applies a low pass filter and can be useful for removing noise.
/// @param cutoffFrequency Fraction of the Maximum Frequency of the image. A value of 0.5 implies that the circular mask has a diameter equal to the half width of the FFT image. This parameter is ignored if removeHighFrequencies is set to false. It must be a strictly positive value, lower than or equal to 1.
/// @param transitionBand Fraction of the cut-off frequency to define the width of the transition band. This parameter is ignored if removeHighFrequencies is set to false. It must be a positive or null value, lower than or equal to 1.
/// @param outputImage The output image. The output image characteristics are forced to the same as the input image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
radialFrequencyFilter2d( std::shared_ptr< iolink::ImageView > inputImage,
                         RadialFrequencyFilter2d::FilterMode filterMode,
                         RadialFrequencyFilter2d::BackgroundModel backgroundModel,
                         bool removeHighFrequencies,
                         double cutoffFrequency,
                         double transitionBand,
                         std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
