/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Creates synthetic spheres in a binary image.
class IMAGEDEV_CPP_API RandomSphereImage3d final : public GenericAlgorithm
{
public:
    /// The distribution type of the sphere radii.
    enum Distribution
    {
    /// The radius distribution is uniform (each radius of the range has an equiprobability to occur).
        UNIFORM = 0,
    /// The radius distribution is Gaussian (the mean radius of the range is much more represented than the minimum and maximum radii).
        GAUSSIAN
    };

    // Command constructor.
    RandomSphereImage3d();


    /// Gets the imageSize parameter.
    /// The X, Y, and Z size in voxels of the output image.
    iolink::Vector3i32 imageSize() const;
    /// Sets the imageSize parameter.
    /// The X, Y, and Z size in voxels of the output image.
    void setImageSize( const iolink::Vector3i32& imageSize );

    /// Gets the sphereNumber parameter.
    /// The number of spheres to generate.
    int32_t sphereNumber() const;
    /// Sets the sphereNumber parameter.
    /// The number of spheres to generate.
    void setSphereNumber( const int32_t& sphereNumber );

    /// Gets the distribution parameter.
    /// The distribution type of the sphere radii.
    RandomSphereImage3d::Distribution distribution() const;
    /// Sets the distribution parameter.
    /// The distribution type of the sphere radii.
    void setDistribution( const RandomSphereImage3d::Distribution& distribution );

    /// Gets the radiusRange parameter.
    /// The minimum and maximum radius in voxels of the spheres (for Uniform distribution) or mean and standard deviation (for gaussian distribution).
    iolink::Vector2i32 radiusRange() const;
    /// Sets the radiusRange parameter.
    /// The minimum and maximum radius in voxels of the spheres (for Uniform distribution) or mean and standard deviation (for gaussian distribution).
    void setRadiusRange( const iolink::Vector2i32& radiusRange );

    /// Gets the distance parameter.
    /// The minimum distance between two sphere centers.
    int32_t distance() const;
    /// Sets the distance parameter.
    /// The minimum distance between two sphere centers.
    void setDistance( const int32_t& distance );

    /// Gets the outputBinaryImage parameter.
    /// The output 3D binary image.
    std::shared_ptr< iolink::ImageView > outputBinaryImage() const;
    /// Sets the outputBinaryImage parameter.
    /// The output 3D binary image.
    void setOutputBinaryImage( std::shared_ptr< iolink::ImageView > outputBinaryImage );

    // Method to launch the command.
    void execute();

};

/// Creates synthetic spheres in a binary image.
/// @param imageSize The X, Y, and Z size in voxels of the output image.
/// @param sphereNumber The number of spheres to generate.
/// @param distribution The distribution type of the sphere radii.
/// @param radiusRange The minimum and maximum radius in voxels of the spheres (for Uniform distribution) or mean and standard deviation (for gaussian distribution).
/// @param distance The minimum distance between two sphere centers.
/// @param outputBinaryImage The output 3D binary image.
/// @return Returns the outputBinaryImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
randomSphereImage3d( const iolink::Vector3i32& imageSize,
                     int32_t sphereNumber,
                     RandomSphereImage3d::Distribution distribution,
                     const iolink::Vector2i32& radiusRange,
                     int32_t distance,
                     std::shared_ptr< iolink::ImageView > outputBinaryImage = nullptr );
} // namespace imagedev
