/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Resamples an image after applying a transformation described by a displacement field.
class IMAGEDEV_CPP_API ResampleElastic2d final : public GenericAlgorithm
{
public:
    /// The type of extrapolation used to fill areas with no antecedent in the input image.
    enum ExtrapolationType
    {
    /// A constant value is used for extrapolation.
        PADDING_VALUE = 0,
    /// The value of the nearest point is used for extrapolation.
        NEAREST_VALUE
    };
    /// The type of interpolation used.
    enum InterpolationType
    {
    /// The value of the intensity is linearly interpolated in all directions.
        NEAREST_NEIGHBOR = 0,
    /// The value of the intensity is equal to the nearest intensity.
        LINEAR
    };

    // Command constructor.
    ResampleElastic2d();


    /// Gets the inputImage parameter.
    /// The grayscale input image to resample. It can have integer or floating point data type and several spectral channels.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The grayscale input image to resample. It can have integer or floating point data type and several spectral channels.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputDisplacementField parameter.
    /// The displacement field mapping the output pixels to the input pixels. It must have exactly two channels and 32-bit float data type. Each pixel of this image image contains two values: a displacement in the X direction (first channel) and one in the Y direction (second channel).
    std::shared_ptr< iolink::ImageView > inputDisplacementField() const;
    /// Sets the inputDisplacementField parameter.
    /// The displacement field mapping the output pixels to the input pixels. It must have exactly two channels and 32-bit float data type. Each pixel of this image image contains two values: a displacement in the X direction (first channel) and one in the Y direction (second channel).
    void setInputDisplacementField( std::shared_ptr< iolink::ImageView > inputDisplacementField );

    /// Gets the extrapolationType parameter.
    /// The type of extrapolation used to fill areas with no antecedent in the input image.
    ResampleElastic2d::ExtrapolationType extrapolationType() const;
    /// Sets the extrapolationType parameter.
    /// The type of extrapolation used to fill areas with no antecedent in the input image.
    void setExtrapolationType( const ResampleElastic2d::ExtrapolationType& extrapolationType );

    /// Gets the interpolationType parameter.
    /// The type of interpolation used.
    ResampleElastic2d::InterpolationType interpolationType() const;
    /// Sets the interpolationType parameter.
    /// The type of interpolation used.
    void setInterpolationType( const ResampleElastic2d::InterpolationType& interpolationType );

    /// Gets the origin parameter.
    /// The origin of the output image, in world coordinates.
    iolink::Vector2d origin() const;
    /// Sets the origin parameter.
    /// The origin of the output image, in world coordinates.
    void setOrigin( const iolink::Vector2d& origin );

    /// Gets the voxelSize parameter.
    /// The pixel size in world coordinates for the X and Y directions of the output image.
    iolink::Vector2d voxelSize() const;
    /// Sets the voxelSize parameter.
    /// The pixel size in world coordinates for the X and Y directions of the output image.
    void setVoxelSize( const iolink::Vector2d& voxelSize );

    /// Gets the imageDimensions parameter.
    /// The size in pixels of the output image in the X and Y directions.
    iolink::Vector2u32 imageDimensions() const;
    /// Sets the imageDimensions parameter.
    /// The size in pixels of the output image in the X and Y directions.
    void setImageDimensions( const iolink::Vector2u32& imageDimensions );

    /// Gets the paddingValue parameter.
    /// The padding value used to fill areas with no antecedent in the input image. This parameter is ignored if extrapolation type is not set to PADDING_VALUE.
    double paddingValue() const;
    /// Sets the paddingValue parameter.
    /// The padding value used to fill areas with no antecedent in the input image. This parameter is ignored if extrapolation type is not set to PADDING_VALUE.
    void setPaddingValue( const double& paddingValue );

    /// Gets the outputImage parameter.
    /// The resampled output image. Its resolution and bounding box are defined by the origin, voxelSize, and imageDimensions parameters. Its type is forced to floating point precision (32-bit).
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The resampled output image. Its resolution and bounding box are defined by the origin, voxelSize, and imageDimensions parameters. Its type is forced to floating point precision (32-bit).
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Resamples an image after applying a transformation described by a displacement field.
/// @param inputImage The grayscale input image to resample. It can have integer or floating point data type and several spectral channels.
/// @param inputDisplacementField The displacement field mapping the output pixels to the input pixels. It must have exactly two channels and 32-bit float data type. Each pixel of this image image contains two values: a displacement in the X direction (first channel) and one in the Y direction (second channel).
/// @param extrapolationType The type of extrapolation used to fill areas with no antecedent in the input image.
/// @param interpolationType The type of interpolation used.
/// @param origin The origin of the output image, in world coordinates.
/// @param voxelSize The pixel size in world coordinates for the X and Y directions of the output image.
/// @param imageDimensions The size in pixels of the output image in the X and Y directions.
/// @param paddingValue The padding value used to fill areas with no antecedent in the input image. This parameter is ignored if extrapolation type is not set to PADDING_VALUE.
/// @param outputImage The resampled output image. Its resolution and bounding box are defined by the origin, voxelSize, and imageDimensions parameters. Its type is forced to floating point precision (32-bit).
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
resampleElastic2d( std::shared_ptr< iolink::ImageView > inputImage,
                   std::shared_ptr< iolink::ImageView > inputDisplacementField,
                   ResampleElastic2d::ExtrapolationType extrapolationType,
                   ResampleElastic2d::InterpolationType interpolationType,
                   const iolink::Vector2d& origin,
                   const iolink::Vector2d& voxelSize,
                   const iolink::Vector2u32& imageDimensions,
                   double paddingValue,
                   std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
