/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a linear scaling of the gray level values of an image.
class IMAGEDEV_CPP_API RescaleIntensity final : public GenericAlgorithm
{
public:
    /// The output image data type.
    enum OutputType
    {
    /// The output image data type is 1 byte depth. Its possible intensities are unsigned integer from 0 to 255.
        UNSIGNED_INTEGER_8_BIT = 0,
    /// The output image data type is 2 bytes depth. Its possible intensities are unsigned integer from 0 to 65,535.
        UNSIGNED_INTEGER_16_BIT,
    /// The output image data type is 4 bytes depth. Its possible intensities are signed integer from 0 to 4,294,967,295.
        UNSIGNED_INTEGER_32_BIT,
    /// The output image data type is 1 byte depth. Its possible intensities are signed integer from -128 to 127.
        SIGNED_INTEGER_8_BIT,
    /// The output image data type is 2 bytes depth. Its possible intensities are signed integer from -32,768 to 32,767.
        SIGNED_INTEGER_16_BIT,
    /// The output image data type is 4 bytes depth. Its possible intensities are signed integer from -2,147,483,648 to 2,147,483,647.
        SIGNED_INTEGER_32_BIT,
    /// The output image data type is 4 bytes depth. Its possible intensities are unsigned floating-point from -3.4E38 to 3.4E38.
        FLOAT_32_BIT,
    /// The output image data type is 8 bytes depth. Its possible intensities are unsigned floating-point from -1.7E308 to 1.7E308.
        FLOAT_64_BIT,
    /// The output image has the same type as the input image.
        SAME_AS_INPUT
    };
    /// The way to define the input intensity range.
    enum RangeMode
    {
    /// The input intensity range is between the minimum and the maximum of the image intensities.
        MIN_MAX = 0,
    /// The input intensity range is between user-defined bounds [a,b].
        OTHER,
    /// The input intensity range is between two user-defined percentiles of the image histogram.
        PERCENTILE
    };

    // Command constructor.
    RescaleIntensity();


    /// Gets the inputImage parameter.
    /// The input image to normalize.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image to normalize.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the outputType parameter.
    /// The output image data type.
    RescaleIntensity::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The output image data type.
    void setOutputType( const RescaleIntensity::OutputType& outputType );

    /// Gets the rangeMode parameter.
    /// The way to define the input intensity range.
    RescaleIntensity::RangeMode rangeMode() const;
    /// Sets the rangeMode parameter.
    /// The way to define the input intensity range.
    void setRangeMode( const RescaleIntensity::RangeMode& rangeMode );

    /// Gets the percentageRange parameter.
    /// The low and high histogram percentile to use when the range mode is set to PERCENTILE (in percent). This parameter is ignored with other range modes.
    iolink::Vector2d percentageRange() const;
    /// Sets the percentageRange parameter.
    /// The low and high histogram percentile to use when the range mode is set to PERCENTILE (in percent). This parameter is ignored with other range modes.
    void setPercentageRange( const iolink::Vector2d& percentageRange );

    /// Gets the intensityInputRange parameter.
    /// The input intensity range [a, b] to use when the range mode is set to OTHER. This parameter is ignored with other range modes.
    iolink::Vector2d intensityInputRange() const;
    /// Sets the intensityInputRange parameter.
    /// The input intensity range [a, b] to use when the range mode is set to OTHER. This parameter is ignored with other range modes.
    void setIntensityInputRange( const iolink::Vector2d& intensityInputRange );

    /// Gets the intensityOutputRange parameter.
    /// The output intensity range.
    iolink::Vector2d intensityOutputRange() const;
    /// Sets the intensityOutputRange parameter.
    /// The output intensity range.
    void setIntensityOutputRange( const iolink::Vector2d& intensityOutputRange );

    /// Gets the outputImage parameter.
    /// The output image. Its dimensions are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its dimensions are forced to the same values as the input.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Performs a linear scaling of the gray level values of an image.
/// @param inputImage The input image to normalize.
/// @param outputType The output image data type.
/// @param rangeMode The way to define the input intensity range.
/// @param percentageRange The low and high histogram percentile to use when the range mode is set to PERCENTILE (in percent). This parameter is ignored with other range modes.
/// @param intensityInputRange The input intensity range [a, b] to use when the range mode is set to OTHER. This parameter is ignored with other range modes.
/// @param intensityOutputRange The output intensity range.
/// @param outputImage The output image. Its dimensions are forced to the same values as the input.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
rescaleIntensity( std::shared_ptr< iolink::ImageView > inputImage,
                  RescaleIntensity::OutputType outputType,
                  RescaleIntensity::RangeMode rangeMode,
                  const iolink::Vector2d& percentageRange,
                  const iolink::Vector2d& intensityInputRange,
                  const iolink::Vector2d& intensityOutputRange,
                  std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
