/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Applies a rotation of a given angle on a two-dimensional image around a given point or the center of the image.
class IMAGEDEV_CPP_API Rotate2d final : public GenericAlgorithm
{
public:
    /// The way to define the rotation center
    enum CenterMode
    {
    /// The rotation center is the center of the input image.
        IMAGE_CENTER = 0,
    /// The rotation center is user-defined. It can be set with the rotationCenter parameter.
        OTHER
    };
    /// The interpolation mode. Method used to calculate the intensity of each pixel in the result image.
    enum InterpolationType
    {
    /// Assign the gray level of the nearest pixel.
        NEAREST_NEIGHBOR = 0,
    /// Assign the bilinear interpolation from the four nearest pixels.
        LINEAR
    };

    // Command constructor.
    Rotate2d();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the rotationAngle parameter.
    /// The angle of the rotation in degrees.
    double rotationAngle() const;
    /// Sets the rotationAngle parameter.
    /// The angle of the rotation in degrees.
    void setRotationAngle( const double& rotationAngle );

    /// Gets the centerMode parameter.
    /// The way to define the rotation center
    Rotate2d::CenterMode centerMode() const;
    /// Sets the centerMode parameter.
    /// The way to define the rotation center
    void setCenterMode( const Rotate2d::CenterMode& centerMode );

    /// Gets the rotationCenter parameter.
    /// The rotation center coordinates. This parameter is ignored in IMAGE_CENTER mode.
    iolink::Vector2d rotationCenter() const;
    /// Sets the rotationCenter parameter.
    /// The rotation center coordinates. This parameter is ignored in IMAGE_CENTER mode.
    void setRotationCenter( const iolink::Vector2d& rotationCenter );

    /// Gets the interpolationType parameter.
    /// The interpolation mode. Method used to calculate the intensity of each pixel in the result image.
    Rotate2d::InterpolationType interpolationType() const;
    /// Sets the interpolationType parameter.
    /// The interpolation mode. Method used to calculate the intensity of each pixel in the result image.
    void setInterpolationType( const Rotate2d::InterpolationType& interpolationType );

    /// Gets the outputResizing parameter.
    /// The output shape policy. If set to True, the output image is resized to ensure that all input data is visible. Otherwise, the output shape remains the same as the input shape.
    bool outputResizing() const;
    /// Sets the outputResizing parameter.
    /// The output shape policy. If set to True, the output image is resized to ensure that all input data is visible. Otherwise, the output shape remains the same as the input shape.
    void setOutputResizing( const bool& outputResizing );

    /// Gets the paddingValue parameter.
    /// The background value for output voxels which are outside the input image domain.
    double paddingValue() const;
    /// Sets the paddingValue parameter.
    /// The background value for output voxels which are outside the input image domain.
    void setPaddingValue( const double& paddingValue );

    /// Gets the outputImage parameter.
    /// The output image. Its type is forced to the same values as the input. Its dimensions are identical to or greater than the input image, according to the outputResizing parameter.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its type is forced to the same values as the input. Its dimensions are identical to or greater than the input image, according to the outputResizing parameter.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Applies a rotation of a given angle on a two-dimensional image around a given point or the center of the image.
/// @param inputImage The input image.
/// @param rotationAngle The angle of the rotation in degrees.
/// @param centerMode The way to define the rotation center
/// @param rotationCenter The rotation center coordinates. This parameter is ignored in IMAGE_CENTER mode.
/// @param interpolationType The interpolation mode. Method used to calculate the intensity of each pixel in the result image.
/// @param outputResizing The output shape policy. If set to True, the output image is resized to ensure that all input data is visible. Otherwise, the output shape remains the same as the input shape.
/// @param paddingValue The background value for output voxels which are outside the input image domain.
/// @param outputImage The output image. Its type is forced to the same values as the input. Its dimensions are identical to or greater than the input image, according to the outputResizing parameter.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
rotate2d( std::shared_ptr< iolink::ImageView > inputImage,
          double rotationAngle,
          Rotate2d::CenterMode centerMode,
          const iolink::Vector2d& rotationCenter,
          Rotate2d::InterpolationType interpolationType,
          bool outputResizing,
          double paddingValue,
          std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
