/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Applies a three-dimensional shear transformation along a user-defined direction.
class IMAGEDEV_CPP_API Shearing3d final : public GenericAlgorithm
{
public:
    /// The axis orthogonal to the slices that are shifted. If the z-axis is selected as the slice axis, the xy slices are shifted along the x or y direction.
    enum Axis
    {
    /// The slice axis is the x-axis, the yz slices are shifted.
        X_AXIS = 0,
    /// The slice axis is the y-axis, the xz slices are shifted.
        Y_AXIS,
    /// The slice axis is the z-axis, the xy slices are shifted.
        Z_AXIS
    };
    /// This parameter defines how to calculate the voxel size of the output volume.
    enum OutputSpacingMode
    {
    /// Set the output spacing to the same values as the inputvoxel size. However, the output may differ slightly due to numerical sampling and rounding.
        SAME_AS_INPUT = 0,
    /// Use the outputSpacing parameter to set the output voxel size. It may be useful to change it when the input has an anisotropic voxel size, for instance when the z-axis has a greater spacing than x and y. The output voxel size uses the same unit as the input image.
        OTHER
    };

    // Command constructor.
    Shearing3d();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the sliceAxis parameter.
    /// The axis orthogonal to the slices that are shifted. If the z-axis is selected as the slice axis, the xy slices are shifted along the x or y direction.
    Shearing3d::Axis sliceAxis() const;
    /// Sets the sliceAxis parameter.
    /// The axis orthogonal to the slices that are shifted. If the z-axis is selected as the slice axis, the xy slices are shifted along the x or y direction.
    void setSliceAxis( const Shearing3d::Axis& sliceAxis );

    /// Gets the shiftAxis parameter.
    /// The direction toward which the slices are shifted. If the z-axis is selected as the slice axis and the y-axis as the shift axis, the xy-slices are shifted along the y direction.
    Shearing3d::Axis shiftAxis() const;
    /// Sets the shiftAxis parameter.
    /// The direction toward which the slices are shifted. If the z-axis is selected as the slice axis and the y-axis as the shift axis, the xy-slices are shifted along the y direction.
    void setShiftAxis( const Shearing3d::Axis& shiftAxis );

    /// Gets the angle parameter.
    /// The shear angle. It represents the angle in degrees between the initial slice axis and the shifted axis.
    double angle() const;
    /// Sets the angle parameter.
    /// The shear angle. It represents the angle in degrees between the initial slice axis and the shifted axis.
    void setAngle( const double& angle );

    /// Gets the sliceAxisOrigin parameter.
    /// The slice index along the slice axis which defines the origin of the transformation, that is the plane that is not shifted by the shearing transform.
    double sliceAxisOrigin() const;
    /// Sets the sliceAxisOrigin parameter.
    /// The slice index along the slice axis which defines the origin of the transformation, that is the plane that is not shifted by the shearing transform.
    void setSliceAxisOrigin( const double& sliceAxisOrigin );

    /// Gets the outputSpacingMode parameter.
    /// This parameter defines how to calculate the voxel size of the output volume.
    Shearing3d::OutputSpacingMode outputSpacingMode() const;
    /// Sets the outputSpacingMode parameter.
    /// This parameter defines how to calculate the voxel size of the output volume.
    void setOutputSpacingMode( const Shearing3d::OutputSpacingMode& outputSpacingMode );

    /// Gets the outputSpacing parameter.
    /// The voxel size of the output volume along the x, y, and z axes, expressed in the same unit as the input image. It is used only when the outputSpacingMode parameter is set to OTHER. In this case, these values must be chosen carefully, ensuring consistency with those of the input volume and the unit used. A value that is too large may generate an output volume that is empty or consists of few voxels, while a value that is too small may lead to memory overflow. This parameter is not used when the spacing mode is set to SAME_AS_INPUT
    iolink::Vector3d outputSpacing() const;
    /// Sets the outputSpacing parameter.
    /// The voxel size of the output volume along the x, y, and z axes, expressed in the same unit as the input image. It is used only when the outputSpacingMode parameter is set to OTHER. In this case, these values must be chosen carefully, ensuring consistency with those of the input volume and the unit used. A value that is too large may generate an output volume that is empty or consists of few voxels, while a value that is too small may lead to memory overflow. This parameter is not used when the spacing mode is set to SAME_AS_INPUT
    void setOutputSpacing( const iolink::Vector3d& outputSpacing );

    /// Gets the paddingValue parameter.
    /// The background value for output voxels which are outside the input image domain.
    double paddingValue() const;
    /// Sets the paddingValue parameter.
    /// The background value for output voxels which are outside the input image domain.
    void setPaddingValue( const double& paddingValue );

    /// Gets the outputImage parameter.
    /// The output image. Its type is forced to the same values as the input. The slice and shift axes have dimensions greater than the input image and the third axes has identical dimensions.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its type is forced to the same values as the input. The slice and shift axes have dimensions greater than the input image and the third axes has identical dimensions.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Applies a three-dimensional shear transformation along a user-defined direction.
/// @param inputImage The input image.
/// @param sliceAxis The axis orthogonal to the slices that are shifted. If the z-axis is selected as the slice axis, the xy slices are shifted along the x or y direction.
/// @param shiftAxis The direction toward which the slices are shifted. If the z-axis is selected as the slice axis and the y-axis as the shift axis, the xy-slices are shifted along the y direction.
/// @param angle The shear angle. It represents the angle in degrees between the initial slice axis and the shifted axis.
/// @param sliceAxisOrigin The slice index along the slice axis which defines the origin of the transformation, that is the plane that is not shifted by the shearing transform.
/// @param outputSpacingMode This parameter defines how to calculate the voxel size of the output volume.
/// @param outputSpacing The voxel size of the output volume along the x, y, and z axes, expressed in the same unit as the input image. It is used only when the outputSpacingMode parameter is set to OTHER. In this case, these values must be chosen carefully, ensuring consistency with those of the input volume and the unit used. A value that is too large may generate an output volume that is empty or consists of few voxels, while a value that is too small may lead to memory overflow. This parameter is not used when the spacing mode is set to SAME_AS_INPUT
/// @param paddingValue The background value for output voxels which are outside the input image domain.
/// @param outputImage The output image. Its type is forced to the same values as the input. The slice and shift axes have dimensions greater than the input image and the third axes has identical dimensions.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
shearing3d( std::shared_ptr< iolink::ImageView > inputImage,
            Shearing3d::Axis sliceAxis,
            Shearing3d::Axis shiftAxis,
            double angle,
            double sliceAxisOrigin,
            Shearing3d::OutputSpacingMode outputSpacingMode,
            const iolink::Vector3d& outputSpacing,
            double paddingValue,
            std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
