/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Enhances structures of interest in a two-dimensional image using a multi-scale analysis.
class IMAGEDEV_CPP_API StructureEnhancementFilter2d final : public GenericAlgorithm
{
public:
    /// The type of tensor to compute: gradient tensor or Hessian matrix.
    enum TensorType
    {
    /// The Hessian matrix is extracted.
        HESSIAN = 0,
    /// The gradient structure tensor is extracted.
        GRADIENT
    };
    /// The lightness type of structures to enhance. This parameter is ignored when the gradient tensor type is selected.
    enum Lightness
    {
    /// The bright structures are extracted from a dark background.
        BRIGHT = 0,
    /// The dark structures are extracted from a bright background.
        DARK
    };
    /// The shape of the structures to enhance.
    enum StructureType
    {
    /// The tubular structures are enhanced.
        ROD = 0,
    /// The blob-like structures are enhanced.
        BALL
    };

    // Command constructor.
    StructureEnhancementFilter2d();


    /// Gets the inputImage parameter.
    /// The input image. The image type can be integer or floating point.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image. The image type can be integer or floating point.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the tensorType parameter.
    /// The type of tensor to compute: gradient tensor or Hessian matrix.
    StructureEnhancementFilter2d::TensorType tensorType() const;
    /// Sets the tensorType parameter.
    /// The type of tensor to compute: gradient tensor or Hessian matrix.
    void setTensorType( const StructureEnhancementFilter2d::TensorType& tensorType );

    /// Gets the standardDeviationRange parameter.
    /// The standard deviation of the Gaussian kernel at minimum and maximum scale, in pixel units.
    iolink::Vector2d standardDeviationRange() const;
    /// Sets the standardDeviationRange parameter.
    /// The standard deviation of the Gaussian kernel at minimum and maximum scale, in pixel units.
    void setStandardDeviationRange( const iolink::Vector2d& standardDeviationRange );

    /// Gets the standardDeviationStep parameter.
    /// The standard deviation step. Structures will be detected from minimum to maximum standard deviation at a pitch of this value.
    double standardDeviationStep() const;
    /// Sets the standardDeviationStep parameter.
    /// The standard deviation step. Structures will be detected from minimum to maximum standard deviation at a pitch of this value.
    void setStandardDeviationStep( const double& standardDeviationStep );

    /// Gets the lightness parameter.
    /// The lightness type of structures to enhance. This parameter is ignored when the gradient tensor type is selected.
    StructureEnhancementFilter2d::Lightness lightness() const;
    /// Sets the lightness parameter.
    /// The lightness type of structures to enhance. This parameter is ignored when the gradient tensor type is selected.
    void setLightness( const StructureEnhancementFilter2d::Lightness& lightness );

    /// Gets the structureType parameter.
    /// The shape of the structures to enhance.
    StructureEnhancementFilter2d::StructureType structureType() const;
    /// Sets the structureType parameter.
    /// The shape of the structures to enhance.
    void setStructureType( const StructureEnhancementFilter2d::StructureType& structureType );

    /// Gets the outputImage parameter.
    /// The output image reprensenting the matching score between 0 and 1. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image reprensenting the matching score between 0 and 1. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Enhances structures of interest in a two-dimensional image using a multi-scale analysis.
/// @param inputImage The input image. The image type can be integer or floating point.
/// @param tensorType The type of tensor to compute: gradient tensor or Hessian matrix.
/// @param standardDeviationRange The standard deviation of the Gaussian kernel at minimum and maximum scale, in pixel units.
/// @param standardDeviationStep The standard deviation step. Structures will be detected from minimum to maximum standard deviation at a pitch of this value.
/// @param lightness The lightness type of structures to enhance. This parameter is ignored when the gradient tensor type is selected.
/// @param structureType The shape of the structures to enhance.
/// @param outputImage The output image reprensenting the matching score between 0 and 1. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
structureEnhancementFilter2d( std::shared_ptr< iolink::ImageView > inputImage,
                              StructureEnhancementFilter2d::TensorType tensorType,
                              const iolink::Vector2d& standardDeviationRange,
                              double standardDeviationStep,
                              StructureEnhancementFilter2d::Lightness lightness,
                              StructureEnhancementFilter2d::StructureType structureType,
                              std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
