/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Computes an image of atomic regions that share common characteristics.
class IMAGEDEV_CPP_API Waterpixel final : public GenericAlgorithm
{
public:
    /// The mode for applying the watershed algorithm.
    enum AlgorithmMode
    {
    /// The result is repeatable but slower to compute.
        REPEATABLE = 0,
    /// The result is faster to compute but not repeatable because of asynchronous parallel computation. Since a watershed problem does not generally have a unique solution, two processings of the same image can lead to two different results (both exact).
        FAST
    };
    /// The type of output.
    enum OutputType
    {
    /// The output image is a binary image containing superpixels boundaries.
        LINES = 0,
    /// The output image is a label image representing connected superpixels.
        CONTIGUOUS_REGIONS,
    /// The output image is a label image representing superpixels separated by a line of background.
        SEPARATED_REGIONS
    };

    // Command constructor.
    Waterpixel();


    /// Gets the inputImage parameter.
    /// The input grayscale image highlighting object boundaries, generally a gradient image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input grayscale image highlighting object boundaries, generally a gradient image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the cellSize parameter.
    /// The side size, in pixels, of the square cells used to generate superpixels.
    uint32_t cellSize() const;
    /// Sets the cellSize parameter.
    /// The side size, in pixels, of the square cells used to generate superpixels.
    void setCellSize( const uint32_t& cellSize );

    /// Gets the factor parameter.
    /// The spatial regularization factor.
    uint32_t factor() const;
    /// Sets the factor parameter.
    /// The spatial regularization factor.
    void setFactor( const uint32_t& factor );

    /// Gets the algorithmMode parameter.
    /// The mode for applying the watershed algorithm.
    Waterpixel::AlgorithmMode algorithmMode() const;
    /// Sets the algorithmMode parameter.
    /// The mode for applying the watershed algorithm.
    void setAlgorithmMode( const Waterpixel::AlgorithmMode& algorithmMode );

    /// Gets the outputType parameter.
    /// The type of output.
    Waterpixel::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The type of output.
    void setOutputType( const Waterpixel::OutputType& outputType );

    /// Gets the outputImage parameter.
    /// The output binary or label image. Its dimensions are forced to the same values as the input image. Its type depends on the outputType parameter.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output binary or label image. Its dimensions are forced to the same values as the input image. Its type depends on the outputType parameter.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Computes an image of atomic regions that share common characteristics.
/// @param inputImage The input grayscale image highlighting object boundaries, generally a gradient image.
/// @param cellSize The side size, in pixels, of the square cells used to generate superpixels.
/// @param factor The spatial regularization factor.
/// @param algorithmMode The mode for applying the watershed algorithm.
/// @param outputType The type of output.
/// @param outputImage The output binary or label image. Its dimensions are forced to the same values as the input image. Its type depends on the outputType parameter.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
waterpixel( std::shared_ptr< iolink::ImageView > inputImage,
            uint32_t cellSize,
            uint32_t factor,
            Waterpixel::AlgorithmMode algorithmMode,
            Waterpixel::OutputType outputType,
            std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
