// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>
#include <iolink/Indexer.h>
#include <iolink/VectorX.h>

namespace iolink
{

/**
 * A multi dimensionnal array.
 *
 * This class should not be used as is, you must use its
 * aliases or it will not work. The current aliases are:
 *
 * - Arrayi8
 * - Arrayi16
 * - Arrayi32
 * - Arrayi64
 * - Arrayu8
 * - Arrayu16
 * - Arrayu32
 * - Arrayu64
 * - Arrayf
 * - Arrayd
 *
 * @tparam T The type of element stored in the array.
 */
template <typename T>
class ArrayX
{
public:
  /**
   * The data type of array's elements.
   */
  using ValueType = T;

  /**
   * Create an array with the given shape.
   *
   * The memory layout of the array will be the default one used
   * by IOLink, the Fortran one, in wich the leftmost dimension is
   * the continuous one.
   *
   * @param shape The shape of the array.
   */
  explicit ArrayX(const VectorXu64& shape);

  /**
   * Create an array with a memory layout fitting the given indexer.
   *
   * @param indexer the Indexer instance describing the mmeory layout.
   */
  explicit ArrayX(const Indexer& indexer);

  // The following default implementations are needed
  // for the Python wrapping.

  ArrayX(const ArrayX& other);
  ArrayX& operator=(const ArrayX& other);

  ArrayX(ArrayX&& other) noexcept;
  ArrayX& operator=(ArrayX&& other) noexcept;

  ~ArrayX();

  /**
   * The number of dimensions of the array.
   */
  size_t dimensionCount() const;

  /**
   * Return the shape of the array.
   */
  const VectorXu64& shape() const;

  /**
   * Return the indexer instance used by the array.
   */
  const Indexer& indexer() const;

  /**
   * Get an element of the array.
   *
   * @warning If Array has an empty shape (dimension count = 0), this method
   * has an undefined behavior.
   *
   * @param index The index of the element to get.
   */
  T at(const VectorXu64& index) const;

  /**
   * Set a value to an element of the array.
   *
   * @warning If Array has an empty shape (dimension count = 0), this method
   * has an undefined behavior.
   *
   * @param index The index of the element to set.
   * @param value The value to set at the given index.
   */
  void setAt(const VectorXu64& index, T value);

  /**
   * Change the shape of the array.
   *
   * @param shape The new shape.
   */
  void reshape(const VectorXu64& shape);

  /**
   * Return a pointer to the internal buffer.
   */
  const T* buffer() const;

  /**
   * Return a non-const pointer to the internal buffer.
   */
  T* buffer();

  /**
   * Return a string representation
   */
  std::string toString() const;

  bool operator==(const ArrayX& other) const;

  bool operator!=(const ArrayX& other) const;

private:
  class Private;
  Private* m_private;
};

// ===================== //
// Template declarations //
// ===================== //

extern template class IOLINK_API_IMPORT ArrayX<int8_t>;
extern template class IOLINK_API_IMPORT ArrayX<int16_t>;
extern template class IOLINK_API_IMPORT ArrayX<int32_t>;
extern template class IOLINK_API_IMPORT ArrayX<int64_t>;
extern template class IOLINK_API_IMPORT ArrayX<uint8_t>;
extern template class IOLINK_API_IMPORT ArrayX<uint16_t>;
extern template class IOLINK_API_IMPORT ArrayX<uint32_t>;
extern template class IOLINK_API_IMPORT ArrayX<uint64_t>;
extern template class IOLINK_API_IMPORT ArrayX<float>;
extern template class IOLINK_API_IMPORT ArrayX<double>;

//==================== //
// Aliases declaration //
//==================== //

using ArrayXi8 = ArrayX<int8_t>;    ///< Alias for int8_t arrays
using ArrayXi16 = ArrayX<int16_t>;  ///< Alias for int16_t arrays
using ArrayXi32 = ArrayX<int32_t>;  ///< Alias for int32_t arrays
using ArrayXi64 = ArrayX<int64_t>;  ///< Alias for int64_t arrays
using ArrayXu8 = ArrayX<uint8_t>;   ///< Alias for uint8_t arrays
using ArrayXu16 = ArrayX<uint16_t>; ///< Alias for uint16_t arrays
using ArrayXu32 = ArrayX<uint32_t>; ///< Alias for uint32_t arrays
using ArrayXu64 = ArrayX<uint64_t>; ///< Alias for uint64_t arrays
using ArrayXf = ArrayX<float>;      ///< Alias for float arrays
using ArrayXd = ArrayX<double>;     ///< Alias for double arrays

} // end namespace iolink
