// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <cstdint>

namespace iolink
{

/**
 * Template class to handle a flags system from an enum.
 *
 * The wrapped enum must have the values set to follow a flags logic, flags values
 * must be multiples of two. The enum can also have shorcut values that combines
 * multiple basic values.
 *
 * Setup:
 *
 * @code{.cpp}
 * // our enum to wrap
 * enum class TestEnum
 * {
 *   Value1 = 0x1, // basic value: 001
 *   Value2 = 0x2, // basic value: 010
 *   Value3 = 0x4, // basic value: 100
 *
 *   // Value1 and Value2
 *   Combo1 = 0x3, // shortcut value: 011
 *
 *   // Value1 and Value3
 *   Combo2 = 0x5, // shortcut value: 101
 * };
 *
 * // defines bitwise operators on enum
 * IOLINK_DEFINE_ENUM_BITWISE_OPERATORS(TestEnum)
 *
 * // alias on template type
 * using TestFlagSet = FlagSet<EnumType>;
 * @endcode
 *
 * @tparam EnumType The enum to wrap.
 */
template <typename EnumType>
class FlagSet
{
public:
  /**
   * Initialize with no flags set.
   */
  FlagSet()
      : m_value(0)
  {
  }

  /**
   * Initialize from a value.
   */
  explicit FlagSet(uint64_t value)
      : m_value(value)
  {
  }

  /**
   * Initialize from a value from the wrapped enum type.
   */
  FlagSet(EnumType flags)
      : m_value(static_cast<uint64_t>(flags))
  {
  }

  FlagSet(const FlagSet& other) = default;
  FlagSet& operator=(const FlagSet& other) = default;

  FlagSet(FlagSet&& other) noexcept = default;
  FlagSet& operator=(FlagSet&& other) noexcept = default;

  /**
   * Return the internal value
   */
  uint64_t value() const { return m_value; }

  /**
   * Check if a given set of flags is true.
   */
  bool has(FlagSet flags) const { return (m_value & flags.m_value) == flags.m_value; }

  /**
   * Set the given set of flags to true.
   */
  void add(FlagSet flags) { m_value |= flags.m_value; }

  /**
   * Set the given set of flags to false.
   */
  void remove(FlagSet flags) { m_value &= ~(flags.m_value); }

  bool operator==(FlagSet flags) const { return m_value == flags.m_value; }

  bool operator!=(FlagSet flags) const { return m_value != flags.m_value; }

  // ================= //
  // Bitwise operators //
  // ================= //

  FlagSet& operator|=(FlagSet flags)
  {
    m_value |= flags.m_value;
    return *this;
  }

  FlagSet& operator&=(FlagSet flags)
  {
    m_value &= flags.m_value;
    return *this;
  }

  FlagSet& operator^=(FlagSet flags)
  {
    m_value ^= flags.m_value;
    return *this;
  }

  FlagSet operator|(FlagSet flags) { return flags |= *this; }

  FlagSet operator&(FlagSet flags) { return flags &= *this; }

  FlagSet operator^(FlagSet flags) { return flags ^= *this; }

private:
  uint64_t m_value;
};

/**
 * This Macro must be called to support bitwise operators on an enum.
 *
 * Enables the use of this kind of expressions with enum values:
 *  `SomeEnum::Value1 | SomeEnum::Value2`
 */
#define IOLINK_DEFINE_ENUM_BITWISE_OPERATORS(EnumType)                                                     \
  inline FlagSet<EnumType> operator|(EnumType lhs, EnumType rhs) { return FlagSet<EnumType>(lhs) |= rhs; } \
  inline FlagSet<EnumType> operator&(EnumType lhs, EnumType rhs) { return FlagSet<EnumType>(lhs) &= rhs; } \
  inline FlagSet<EnumType> operator^(EnumType lhs, EnumType rhs) { return FlagSet<EnumType>(lhs) ^= rhs; }

} // end namespace iolink
