// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <memory>

#include <iolink/Matrix.h>
#include <iolink/Vector.h>
#include <iolink/VectorX.h>
#include <iolink/storage/StreamAccess.h>
#include <iolink/view/DataFrameView.h>
#include <iolink/view/ImageView.h>

namespace iolink
{

/**
 * Serialization class which provides methods to serialize and
 * unserialize most of IOLink objects.
 */
class IOLINK_API Serialization
{
public:
  Serialization() = delete;
  Serialization(const Serialization& other) = delete;            // copy constructor
  Serialization(Serialization&& other) = delete;                 // move constructor
  Serialization& operator=(const Serialization& other) = delete; // copy assignment
  Serialization& operator=(Serialization&& other) = delete;      // move assignment

  //================================================
  // ImageView methods

  // encode
  /**
   * Factory to serialize an ImageView into a stream.
   * @param image ImageView to serialize
   * @param outStream Stream where data is written
   * @throw Error if ImageView has not READ capacity
   */
  static void encodeImageView(std::shared_ptr<ImageView> image, std::shared_ptr<StreamAccess> outStream);

  //------------
  // decode

  /**
   * Factory to unserialize an ImageView from a stream.
   * @param inStream Stream from where data is read
   * @return a in-memory ImageView built from input Stream
   * @throw Error if IOLink version read from stream is not compatible with current version
   * @throw Error if read magic number is invalid
   * @throw Error if object to decode is not an ImageView
   * @throw Error if decoding issue (synchronization tag not found)
   */
  static std::shared_ptr<ImageView> decodeImageView(std::shared_ptr<StreamAccess> inStream);

  //================================================
  // Array methods

  /**
   * Serialize an Array into a stream.
   *
   * @param src The array to encode.
   * @param dst The stream where the encoded data is written.
   */
  static void encodeArrayXi8(const ArrayXi8& src, std::shared_ptr<StreamAccess> dst);

  /**
   * @copydoc encodeArrayXi8
   */
  static void encodeArrayXi16(const ArrayXi16& src, std::shared_ptr<StreamAccess> dst);

  /**
   * @copydoc encodeArrayXi8
   */
  static void encodeArrayXi32(const ArrayXi32& src, std::shared_ptr<StreamAccess> dst);

  /**
   * @copydoc encodeArrayXi8
   */
  static void encodeArrayXi64(const ArrayXi64& src, std::shared_ptr<StreamAccess> dst);

  /**
   * @copydoc encodeArrayXi8
   */
  static void encodeArrayXu8(const ArrayXu8& src, std::shared_ptr<StreamAccess> dst);

  /**
   * @copydoc encodeArrayXi8
   */
  static void encodeArrayXu16(const ArrayXu16& src, std::shared_ptr<StreamAccess> dst);

  /**
   * @copydoc encodeArrayXi8
   */
  static void encodeArrayXu32(const ArrayXu32& src, std::shared_ptr<StreamAccess> dst);

  /**
   * @copydoc encodeArrayXi8
   */
  static void encodeArrayXu64(const ArrayXu64& src, std::shared_ptr<StreamAccess> dst);

  /**
   * @copydoc encodeArrayXi8
   */
  static void encodeArrayXf(const ArrayXf& src, std::shared_ptr<StreamAccess> dst);

  /**
   * @copydoc encodeArrayXi8
   */
  static void encodeArrayXd(const ArrayXd& src, std::shared_ptr<StreamAccess> dst);

  /**
   * Decode an array from a stream.
   *
   * @param src The stream to decode from.
   *
   * @return The decoded array.
   */
  static ArrayXi8 decodeArrayXi8(std::shared_ptr<StreamAccess> src);

  /**
   * @copydoc decodeArrayXi8
   */
  static ArrayXi16 decodeArrayXi16(std::shared_ptr<StreamAccess> src);

  /**
   * @copydoc decodeArrayXi8
   */
  static ArrayXi32 decodeArrayXi32(std::shared_ptr<StreamAccess> src);

  /**
   * @copydoc decodeArrayXi8
   */
  static ArrayXi64 decodeArrayXi64(std::shared_ptr<StreamAccess> src);

  /**
   * @copydoc decodeArrayXi8
   */
  static ArrayXu8 decodeArrayXu8(std::shared_ptr<StreamAccess> src);

  /**
   * @copydoc decodeArrayXi8
   */
  static ArrayXu16 decodeArrayXu16(std::shared_ptr<StreamAccess> src);

  /**
   * @copydoc decodeArrayXi8
   */
  static ArrayXu32 decodeArrayXu32(std::shared_ptr<StreamAccess> src);

  /**
   * @copydoc decodeArrayXi8
   */
  static ArrayXu64 decodeArrayXu64(std::shared_ptr<StreamAccess> src);

  /**
   * @copydoc decodeArrayXi8
   */
  static ArrayXf decodeArrayXf(std::shared_ptr<StreamAccess> src);

  /**
   * @copydoc decodeArrayXi8
   */
  static ArrayXd decodeArrayXd(std::shared_ptr<StreamAccess> src);

  //================================================
  // Vector methods

  // 2D
  //------------
  // encode
  /**
   * Factory to serialize a Vector2 into a stream.
   * @param vect Vector2 to serialize
   * @param outStream Stream where data is written
   */
  static void encodeVector2u8(const Vector2u8& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector2u8*/
  static void encodeVector2u16(const Vector2u16& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector2u8*/
  static void encodeVector2u32(const Vector2u32& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector2u8*/
  static void encodeVector2u64(const Vector2u64& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector2u8*/
  static void encodeVector2i8(const Vector2i8& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector2u8*/
  static void encodeVector2i16(const Vector2i16& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector2u8*/
  static void encodeVector2i32(const Vector2i32& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector2u8*/
  static void encodeVector2i64(const Vector2i64& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector2u8*/
  static void encodeVector2f(const Vector2f& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector2u8*/
  static void encodeVector2d(const Vector2d& vect, std::shared_ptr<StreamAccess> outStream);

  //------------
  // decode

  /**
   * Factory to unserialize a Vector2 from a stream.
   * @param inStream Stream from where data is read
   * @return a Vector2 built from input Stream
   * @throw Error if IOLink version read from stream is not compatible with current version
   * @throw Error if read magic number is invalid
   * @throw Error if object to decode is not a Vector
   * @throw Error if data type read from stream is not compatible with requested vector type
   */
  static Vector2u8 decodeVector2u8(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector2u8*/
  static Vector2u16 decodeVector2u16(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector2u8*/
  static Vector2u32 decodeVector2u32(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector2u8*/
  static Vector2u64 decodeVector2u64(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector2u8*/
  static Vector2i8 decodeVector2i8(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector2u8*/
  static Vector2i16 decodeVector2i16(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector2u8*/
  static Vector2i32 decodeVector2i32(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector2u8*/
  static Vector2i64 decodeVector2i64(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector2u8*/
  static Vector2f decodeVector2f(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector2u8*/
  static Vector2d decodeVector2d(std::shared_ptr<StreamAccess> inStream);

  //------------
  // 3D
  //------------
  // encode

  /**
   * Factory to serialize a Vector3 into a stream.
   * @param vect Vector3 to serialize
   * @param outStream Stream where data is written
   */
  static void encodeVector3u8(const Vector3u8& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector3u8*/
  static void encodeVector3u16(const Vector3u16& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector3u8*/
  static void encodeVector3u32(const Vector3u32& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector3u8*/
  static void encodeVector3u64(const Vector3u64& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector3u8*/
  static void encodeVector3i8(const Vector3i8& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector3u8*/
  static void encodeVector3i16(const Vector3i16& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector3u8*/
  static void encodeVector3i32(const Vector3i32& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector3u8*/
  static void encodeVector3i64(const Vector3i64& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector3u8*/
  static void encodeVector3f(const Vector3f& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector3u8*/
  static void encodeVector3d(const Vector3d& vect, std::shared_ptr<StreamAccess> outStream);

  //-------------
  // decode

  /**
   * Method to load a Vector3 which has been serialized into a stream.
   * @param inStream Stream from where data is read
   * @return a Vector3 built from input Stream
   * @throw Error if IOLink version read from stream is not compatible with current version
   * @throw Error if read magic number is invalid
   * @throw Error if object to decode is not a Vector
   * @throw Error if data type read from stream is not compatible with requested vector type
   */
  static Vector3u8 decodeVector3u8(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector3u8*/
  static Vector3u16 decodeVector3u16(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector3u8*/
  static Vector3u32 decodeVector3u32(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector3u8*/
  static Vector3u64 decodeVector3u64(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector3u8*/
  static Vector3i8 decodeVector3i8(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector3u8*/
  static Vector3i16 decodeVector3i16(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector3u8*/
  static Vector3i32 decodeVector3i32(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector3u8*/
  static Vector3i64 decodeVector3i64(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector3u8*/
  static Vector3f decodeVector3f(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector3u8*/
  static Vector3d decodeVector3d(std::shared_ptr<StreamAccess> inStream);

  //------------
  // 4D
  //------------
  // encode

  /**
   * Factory to serialize a Vector4 into a stream.
   * @param vect Vector4 to serialize
   * @param outStream Stream where data is written
   */
  static void encodeVector4u8(const Vector4u8& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector4u8*/
  static void encodeVector4u16(const Vector4u16& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector4u8*/
  static void encodeVector4u32(const Vector4u32& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector4u8*/
  static void encodeVector4u64(const Vector4u64& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector4u8*/
  static void encodeVector4i8(const Vector4i8& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector4u8*/
  static void encodeVector4i16(const Vector4i16& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector4u8*/
  static void encodeVector4i32(const Vector4i32& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector4u8*/
  static void encodeVector4i64(const Vector4i64& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector4u8*/
  static void encodeVector4f(const Vector4f& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVector4u8*/
  static void encodeVector4d(const Vector4d& vect, std::shared_ptr<StreamAccess> outStream);

  //------------
  // decode

  /**
   * Factory to unserialize a Vector4 from a stream.
   * @param inStream Stream from where data is read
   * @return a Vector4 built from input Stream
   * @throw Error if IOLink version read from stream is not compatible with current version
   * @throw Error if read magic number is invalid
   * @throw Error if object to decode is not a Vector
   * @throw Error if data type read from stream is not compatible with requested vector type
   */
  static Vector4u8 decodeVector4u8(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector4u8*/
  static Vector4u16 decodeVector4u16(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector4u8*/
  static Vector4u32 decodeVector4u32(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector4u8*/
  static Vector4u64 decodeVector4u64(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector4u8*/
  static Vector4i8 decodeVector4i8(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector4u8*/
  static Vector4i16 decodeVector4i16(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector4u8*/
  static Vector4i32 decodeVector4i32(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector4u8*/
  static Vector4i64 decodeVector4i64(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector4u8*/
  static Vector4f decodeVector4f(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVector4u8*/
  static Vector4d decodeVector4d(std::shared_ptr<StreamAccess> inStream);

  //================================================
  // VectorX methods
  //------------
  // encode

  /**
   * Factory to serialize a VectorX into a stream.
   * @param vect VectorX to serialize
   * @param outStream Stream where data is written
   */
  static void encodeVectorXu8(const VectorXu8& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVectorXu8*/
  static void encodeVectorXu16(const VectorXu16& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVectorXu8*/
  static void encodeVectorXu32(const VectorXu32& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVectorXu8*/
  static void encodeVectorXu64(const VectorXu64& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVectorXu8*/
  static void encodeVectorXi8(const VectorXi8& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVectorXu8*/
  static void encodeVectorXi16(const VectorXi16& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVectorXu8*/
  static void encodeVectorXi32(const VectorXi32& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVectorXu8*/
  static void encodeVectorXi64(const VectorXi64& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVectorXu8*/
  static void encodeVectorXf(const VectorXf& vect, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeVectorXu8*/
  static void encodeVectorXd(const VectorXd& vect, std::shared_ptr<StreamAccess> outStream);

  //------------
  // decode

  /**
   * Factory to unserialize a VectorX from a stream.
   * @param inStream Stream from where data is read
   * @return a VectorX built from input Stream
   * @throw Error if IOLink version read from stream is not compatible with current version
   * @throw Error if read magic number is invalid
   * @throw Error if object to decode is not a Vector
   * @throw Error if data type read from stream is not compatible with requested vector type
   */
  static VectorXu8 decodeVectorXu8(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVectorXu8*/
  static VectorXu16 decodeVectorXu16(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVectorXu8*/
  static VectorXu32 decodeVectorXu32(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVectorXu8*/
  static VectorXu64 decodeVectorXu64(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVectorXu8*/
  static VectorXi8 decodeVectorXi8(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVectorXu8*/
  static VectorXi16 decodeVectorXi16(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVectorXu8*/
  static VectorXi32 decodeVectorXi32(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVectorXu8*/
  static VectorXi64 decodeVectorXi64(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVectorXu8*/
  static VectorXf decodeVectorXf(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeVectorXu8*/
  static VectorXd decodeVectorXd(std::shared_ptr<StreamAccess> inStream);

  //================================================
  // DataFrameView methods
  //------------
  // encode
  /**
   * Factory to serialize a DataFrameView into a stream.
   * @param dataFrame DataFrameView to serialize
   * @param outStream Stream where data is written
   * @throw Error if DataFrameView has not READ capacity
   */
  static void encodeDataFrameView(std::shared_ptr<DataFrameView> dataFrame, std::shared_ptr<StreamAccess> outStream);

  //------------
  // decode

  /**
   * Factory to unserialize a DataFrameView from a stream.
   * @param inStream Stream from where data is read
   * @return a in-memory DataFrameView built from input Stream
   * @throw Error if IOLink version read from stream is not compatible with current version
   * @throw Error if read magic number is invalid
   * @throw Error if object to decode is not a DataFrameView
   * @throw Error if decoding issue (synchronization tag not found)
   */
  static std::shared_ptr<DataFrameView> decodeDataFrameView(std::shared_ptr<StreamAccess> inStream);

  //================================================
  // Matrix methods
  //------------
  // encode

  /**
   * Factory to serialize a Matrix into a stream.
   * @param mat Matrix to serialize
   * @param outStream Stream where data is written
   */
  static void encodeMatrix3f(const Matrix3f& mat, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeMatrix3f*/
  static void encodeMatrix4f(const Matrix4f& mat, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeMatrix3f*/
  static void encodeMatrix3d(const Matrix3d& mat, std::shared_ptr<StreamAccess> outStream);
  /** @copydoc encodeMatrix3f*/
  static void encodeMatrix4d(const Matrix4d& mat, std::shared_ptr<StreamAccess> outStream);

  //------------
  // decode

  /**
   * Factory to unserialize a Matrix from a stream.
   * @param inStream Stream from where data is read
   * @return a Matrix built from input Stream
   * @throw Error if IOLink version read from stream is not compatible with current version
   * @throw Error if read magic number is invalid
   * @throw Error if object to decode is not a Matrix
   * @throw Error if data type or dimension read from stream is not compatible with requested matrix type
   */
  static Matrix3f decodeMatrix3f(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeMatrix3f*/
  static Matrix4f decodeMatrix4f(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeMatrix3f*/
  static Matrix3d decodeMatrix3d(std::shared_ptr<StreamAccess> inStream);
  /** @copydoc decodeMatrix3f*/
  static Matrix4d decodeMatrix4d(std::shared_ptr<StreamAccess> inStream);
};

} // namespace iolink
