// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <memory>
#include <string>

#include <iolink/IOLinkAPI.h>
#include <iolink/storage/RandomAccess.h>
#include <iolink/storage/StreamAccess.h>

namespace iolink
{

/**
 * Used to request capabilities for accessors
 */
enum class AccessCapabilities
{
  READ_ONLY = 0x1,  // 0001
  WRITE_ONLY = 0x2, // 0010
  READ_WRITE = 0x3, // 0011
};

/**
 * Describes a data source, that manages and opens DataAccess instances.
 */
class IOLINK_INTERFACE_API DataStorage
{
public:
  /**
   * Open a stream-like accessor to a given resource.
   *
   * @param resourceId The resource name
   * @param capabilities The capabilities you want the accessor to support
   *
   * @return The opened stream accessor.
   *
   * @throw Error if the storage can not access to the resource.
   * @throw Error if the resource does not support the requested capabilities.
   *
   * @see DataAccess::resourceId for more info on this identifier.
   */
  virtual std::shared_ptr<StreamAccess> openStreamAccess(const std::string& resourceId,
                                                         AccessCapabilities capabilities) = 0;

  /**
   * Open an array-like accessor to a given resource.
   *
   * @param resourceId The resource name
   * @param capabilities The capabilities you want the accessor to support
   *
   * @return The opened random-access accessor.
   *
   * @throw Error if the storage can not access to the resource.
   * @throw Error if the resource does not support the request capabilities.
   *
   * @see DataAccess::resourceId for more info on this identifier.
   */
  virtual std::shared_ptr<RandomAccess> openRandomAccess(const std::string& resourceId,
                                                         AccessCapabilities capabilities) = 0;

  /**
   * Return a string that describe the storage's source of data.
   *
   * Usually has the form of an URI, for example:
   *  - "file:/home/user/folder" for a filesystem storage
   *  - "http://amazing.website.net" for an HTTP storage
   */
  virtual const std::string& source() const = 0;
};

} // namespace iolink
