// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>
#include <iolink/Uri.h>
#include <iolink/storage/HTTPHeaders.h>
#include <iolink/storage/RandomAccess.h>
#include <iolink/storage/StreamAccess.h>

#include <iostream>
#include <map>
#include <memory>

namespace iolink
{

/**
 * A factory to create StreamAccess instances from various ressources.
 */
class IOLINK_API StreamAccessFactory
{
public:
  StreamAccessFactory() = delete;
  StreamAccessFactory(const StreamAccessFactory& other) = delete;            // copy constructor
  StreamAccessFactory(StreamAccessFactory&& other) = delete;                 // move constructor
  StreamAccessFactory& operator=(const StreamAccessFactory& other) = delete; // copy assignment
  StreamAccessFactory& operator=(StreamAccessFactory&& other) = delete;      // move assignment

  /**
   * Create a stream in local memory
   * @param size memory size (in byte) to allocate for the stream.
   * @return a MemoryStreamAccess with all capabilities (`read`, `write` and `seek`)
   */
  static std::shared_ptr<StreamAccess> allocate(size_t size = 0);

  /**
   * Load a stream into local memory
   * @param streamAccess stream with at least `read` capability to load in memory
   * @return a MemoryStreamAccess with all capabilities (`read`, `write` and `seek`)
   * @throw InvalidArgument if input streamAccess is null
   * @throw Error if the input streamAccess has no read or seek capability
   */
  static std::shared_ptr<StreamAccess> copyInMemory(std::shared_ptr<StreamAccess> streamAccess);

  /**
   * Create a stream storage from a standard input stream.
   * @param stream Standard Input Stream on which a bridge is created
   * @return a StreamAccess object which can eventually provide `seek` capability (see StreamAccessCapability)
   */
  static std::shared_ptr<StreamAccess> fromStdStream(std::shared_ptr<std::istream> stream);

  /**
   * Create a stream storage from a standard output stream.
   * @param stream Standard Output Stream on which a bridge is created
   * @return a StreamAccess object which can eventually provide `seek` capability (see StreamAccessCapability)
   */
  static std::shared_ptr<StreamAccess> fromStdStream(std::shared_ptr<std::ostream> stream);

  /**
   * Create a stream storage from a standard I/O stream.
   * @param stream Standard I/O stream from which a StreamAccess is bridge is created
   * @return a StreamAccess with `read`, `write` and `seek` capabilities (see StreamAccessCapability)
   */
  static std::shared_ptr<StreamAccess> fromStdStream(std::shared_ptr<std::iostream> stream);

  /**
   * Create a standard Input stream from a StreamAccess
   * @param streamAccess stream with `read` capability from which a bridge is created
   * @return a standard Input Stream
   * @throw InvalidArgument if input streamAccess is null
   */
  static std::shared_ptr<std::istream> toStdIStream(std::shared_ptr<StreamAccess> streamAccess);

  /**
   * Create a standard Output stream from a StreamAccess
   * @param streamAccess StreamAccess with `write` capability from which a bridge is created
   * @throw InvalidArgument if input streamAccess is null
   * @return a standard Output Stream
   */
  static std::shared_ptr<std::ostream> toStdOStream(std::shared_ptr<StreamAccess> streamAccess);

  /**
   * Create a standard Input/Output stream from a StreamAccess
   * @param streamAccess stream with `read` and `write` and `seek` capability from which a bridge is created
   * @throw InvalidArgument if input streamAccess is null
   * @throw InvalidArgument if `read`, `write`, or `seek` capabilities is missing
   * @return a standard I/O Stream
   */
  static std::shared_ptr<std::iostream> toStdStream(std::shared_ptr<StreamAccess> streamAccess);

  /**
   * Create a stream that can read data from a file.
   * @param filePath path toward the file which it's tried to access for reading
   * @return a StreamAccess object which can eventually provide `seek` capability (see StreamAccessCapability)
   */
  static std::shared_ptr<StreamAccess> openFileRead(const std::string& filePath);

  /**
   * Create a stream that can write data to a file.
   * @param filePath path toward the file which it's tried to access for writing
   * @return a StreamAccess object which can eventually provide `seek` capability (see StreamAccessCapability)
   */
  static std::shared_ptr<StreamAccess> openFileWrite(const std::string& filePath);

  /**
   * Create a stream that can read and write data from/to a file.
   * @param filePath path toward the file which it's tried to access for reading and writing
   * @return a StreamAccess object which can provide `read`, `write` and eventually `seek` capability (see
   * StreamAccessCapability)
   */
  static std::shared_ptr<StreamAccess> openFile(const std::string& filePath);

  /**
   * Create a stream that can write data to an URI.
   *
   * If the URI scheme is 'file': calls openFileWrite.
   * If the URI scheme is 'http(s)': calls openHTTPWrite, using the POST method and no headers.
   * @param uri URI to open for writing
   * @return a StreamAccess object which can eventually provide `seek` capability (see StreamAccessCapability)
   */
  static std::shared_ptr<StreamAccess> openURIWrite(const Uri& uri);

  /**
   * Create a stream that can post data to an HTTP URI.
   * HTTP server must support chunked transfer encoding.
   *
   * @param uri The HTTP(S) URI to post to.
   * @param method The HTTP request method.
   * @param headers The HTTP request headers.
   * @return a StreamAccess object which can eventually provide `seek` capability (see StreamAccessCapability)
   */
  static std::shared_ptr<StreamAccess> openHTTPWrite(const Uri& uri,
                                                     const std::string& method,
                                                     const HTTPHeaders& headers);

  /**
   * Create a StreamAccess that will delegate to a RandomAccess.
   * @param randomAccess RandomAccess from which a StreamAccess is created
   * @return a StreamAccess object which can provide `read`, `write` and `seek` capability (see StreamAccessCapability)
   * @throw InvalidArgument if input randomAccess is null
   * @throw Error if the input randomAccess has nor read and write capability
   */
  static std::shared_ptr<StreamAccess> fromRandomAccess(std::shared_ptr<RandomAccess> randomAccess);
};

} // end namespace iolink