// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <string>

#include <iolink/DataType.h>
#include <iolink/FlagSet.h>
#include <iolink/IOLinkAPI.h>
#include <iolink/VectorX.h>
#include <iolink/view/View.h>

namespace iolink
{

/**
 * Capabilities of dataframes instances.
 *
 * @see DataFrameView
 */
enum class DataFrameCapability
{
  READ = 0x1,    ///< Ability to read column data
  WRITE = 0x2,   ///< Ability to write column data
  RESHAPE = 0x4, ///< Ability to add or remove columns and rows
};

extern template class IOLINK_API_IMPORT FlagSet<DataFrameCapability>;

/**
 * FlagSet to handle DataFrameView capabilities
 */
using DataFrameCapabilitySet = FlagSet<DataFrameCapability>;

// Define bitwise operators for DataFrameCapability
IOLINK_DEFINE_ENUM_BITWISE_OPERATORS(DataFrameCapability)

/**
 * View to model tabular data.
 *
 * This view is composed of a set of columns, that have a name and a data type.
 * A column's name is unique, and columns in a dataframe can have different data
 * types.
 *
 * Each column has the same number of elements.
 * The aggregate of all n-th elements of columns are a row of the dataframe.
 */
class IOLINK_API DataFrameView : public View
{
public:
  /**
   * Return the view's capabilities
   *
   * @see DataFrameCapability
   */
  virtual DataFrameCapabilitySet capabilities() const = 0;

  /**
   * Return the shape of the data frame.
   *
   * First element being the number of columns, and the second, the number of rows.
   */
  virtual const Vector2u64& shape() const = 0;

  /**
   * Get the name of a column from its index.
   *
   * Column indices go from 0 to shape[0] - 1.
   *
   * @param index The index of the column.
   *
   * @throw Error If index is out of range.
   */
  virtual const std::string& columnName(size_t index) const = 0;

  /**
   * Get the index of a column from its name.
   *
   * @param name The name of the column.
   *
   * @throw Error If there is no column with that name.
   */
  virtual size_t columnIndex(const std::string& name) const = 0;

  /**
   * Return the data type of the column.
   *
   * @param index The index of the column.
   *
   * @throw Error When index is out of range.
   */
  virtual DataType columnDataType(size_t index) const = 0;

  // ==== //
  // READ //
  // ==== //

  /**
   * Read data from a column.
   *
   * This method requires the READ capability.
   *
   * When reading numeric data, the dst buffer will be filled with continuous
   * values. In the case of strings, it will be a continuous array of string
   * objects.
   *
   * The destination buffer must be allocated by the user. If its size is under
   * the byte count needed to store the request number of elements, a segmentation
   * fault will rise, as there is no boundaries check.
   *
   * @param column The index of the column to read from.
   * @param offset The index of the first element to read.
   * @param count The number of elements to read.
   * @param dst A buffer where the read data will be written.
   *
   * @throw NotImplemented When the view does not support reading.
   * @throw Error When dst is null.
   * @throw Error When the column index is invalid.
   * @throw Error When offset is out of bounds.
   * @throw Error If it is impossible to read the number of elements required.
   */
  virtual void read(size_t column, size_t offset, size_t count, void* dst) const;

  /**
   * @brief Get the value stored in a cell of the data frame.
   *
   * @tparam T The type of the data stored in the cell.
   *
   * @param column The index of the column
   * @param row The index of the row
   *
   * @return The value of the cell at the given index.
   *
   * @throw Error if the column or the cell index is out of bounds.
   */
  template <typename T>
  inline T at(size_t column, size_t row) const
  {
    T value;
    this->read(column, row, 1, &value);
    return value;
  }

  /**
   * @brief Get the value stored in a cell of the data frame.
   *
   * @tparam T The type of the data stored in the cell.
   *
   * @param columnName The name of the column where to get the cell.
   * @param row The index of the row
   *
   * @return The value of the chosen cell.
   *
   * @throw Error if the column name does not exist in the data frame.
   * @throw Error if the row is out of bounds.
   */
  template <typename T>
  inline T at(const std::string& columnName, size_t row) const
  {
    return this->at<T>(this->columnIndex(columnName), row);
  }

  /**
   * @copydoc ColumnUnitExtension::unit
   */
  const std::string& unit(size_t columnIndex);

  // ===== //
  // WRITE //
  // ===== //

  /**
   * Write data into a column.
   *
   * This method requires the WRITE capability.
   *
   * When writing numeric data, the src buffer must be filled with continuous
   * values. In the case of strings, the source must use a continuous array of
   * string objects.
   *
   * The source buffer must be allocated by the user. If its size is under
   * the byte count needed to store the request number of elements, a segmentation
   * fault will rise, as there is no boundaries check.
   *
   * @param column The index of the column to write into.
   * @param offset The index of the first element to write.
   * @param count The number of element to write.
   * @param src A buffer whose content will be written in the column.
   *
   * @throw NotImplemented When the view does not support writing.
   * @throw Error When src is null.
   * @throw Error When the column index is invalid.
   * @throw Error When offset is out of bounds.
   * @throw Error If it is impossible to write the number of elements required.
   */
  virtual void write(size_t column, size_t offset, size_t count, const void* src);

  /**
   * @brief Set the element value in a cell of the data frame.
   *
   * @tparam T The type of the data stored in the cell.
   *
   * @param column The column of the cell to set.
   * @param row The row of the cell to set.
   * @param value The value to store in the cell.
   *
   * @throw Error if the column name does not exist in the data frame.
   * @throw Error if the row is out of bounds.
   */
  template <typename T>
  inline void setAt(size_t column, size_t row, T value)
  {
    this->write(column, row, 1, &value);
  }

  /**
   * @brief Set the element value in a cell of the data frame.
   *
   * @tparam T The type of the data stored in the cell.
   *
   * @param columnName The name of the column where to set the cell.
   * @param row The row of the cell to set.
   * @param value The value to store in the cell.
   *
   * @throw Error if the column name does not exist in the data frame.
   * @throw Error if the row is out of bounds.
   */
  template <typename T>
  inline void setAt(const std::string& columnName, size_t row, T value)
  {
    this->setAt<T>(this->columnIndex(columnName), row, value);
  }

  /**
   * @copydoc ColumnUnitExtension::setUnit
   */
  void setUnit(size_t columnIndex, const std::string& unit);

  // ======= //
  // RESHAPE //
  // ======= //

  /**
   * Add a column to the dataframe.
   *
   * This method requires the RESHAPE capability.
   *
   * The new column's data will be undefined after addition.
   *
   * @param name The name of the new column.
   * @param dtype The type of data stored in the column.
   *
   * @throw NotImplemented When the view does not support reshaping.
   * @throw Error if there is already a column with the same name.
   * @throw Error if the implementation does not support the data type.
   */
  virtual void addColumn(const std::string& name, DataType dtype);

  /**
   * Insert rows at the specified position.
   *
   * This method requires the RESHAPE capability.
   *
   * New rows content will be undefined after insertion.
   *
   * @param offset New rows will be inserted before this row. If the offset is
   *               the number of rows in the dataframe, rows will be inserted at
   *               the end.
   * @param count The number of rows to insert.
   *
   * @throw NotImplemented When the view does not support reshaping.
   * @throw Error When offset is out of bounds.
   * @throw Error When it is impossible to add these rows.
   */
  virtual void addRows(size_t offset, size_t count);

  /**
   * Remove a column.
   *
   * This method requires the RESHAPE capability.
   *
   * @param index The index of the column to remove.
   *
   * @throw NotImplemented When the view does not support reshaping.
   * @throw Error If the index is out of bounds.
   * @throw Error When it is impossible to remove that column.
   */
  virtual void removeColumn(size_t index);

  /**
   * Remove a continuous set of rows.
   *
   * This method requires the RESHAPE capability.
   *
   * @param offset The index of the first column to remove.
   * @param count Row count to delete.
   *
   * @throw NotImplemented When the view does not support reshaping.
   * @throw Error When offset is out of bounds.
   * @throw Error When it is impossible to remove these rows.
   */
  virtual void removeRows(size_t offset, size_t count);

  /**
   * Return a string representation
   */
  std::string toString() const;

  /**
   * Method to reshape a DataFrame
   *
   * This method requires the RESHAPE capability.
   *
   * All the dataframe's data is invalidated after this operation.
   *
   * @param shape shape of the allocated dataframe (column count, row count)
   * @param columnNames Array which contains the name for each column
   * @param columnDataTypes Array which contains the data type of each column
   *
   * @warning No check is done on inputs. Pass arguments very carefully.
   * columnNames and columnDataTypes shall have the same element count, and their size
   * must be equal to shape[0].
   * @warning column count and row count cannot be 0.
   */
  void reshape(const Vector2u64& shape, const std::string* columnNames, const DataType* columnDataTypes);
};

} // end namespace iolink
