// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>
#include <iolink/view/ImageView.h>
#include <iolink/view/MultiImageView.h>

#include <memory>

namespace iolink
{

/**
 * This factory is aimed at creating stack of ImageViews.
 */
class IOLINK_API MultiImageViewFactory
{
public:
  MultiImageViewFactory() = delete;
  MultiImageViewFactory(const MultiImageViewFactory& other) = delete;            // copy constructor
  MultiImageViewFactory(MultiImageViewFactory&& other) = delete;                 // move constructor
  MultiImageViewFactory& operator=(const MultiImageViewFactory& other) = delete; // copy assignment
  MultiImageViewFactory& operator=(MultiImageViewFactory&& other) = delete;      // move assignment

  /**
   * Creates an empty MultiImageView
   */
  static std::shared_ptr<MultiImageView> create();

  /**
   * Creates an MultiImageView with given list of frames
   *
   *@throw Error if one ImageView of the given list has invalid properties
   */
  static std::shared_ptr<MultiImageView> create(std::initializer_list<std::shared_ptr<ImageView>> init);

  /**
   * Creates a MultiImageView with given list of frames
   *
   * @param listImages pointer on first array of ImageView to add in MultiImageView
   * @param count elements count in given array
   *@throw Error if one ImageView of the given list has invalid properties
   */
  static std::shared_ptr<MultiImageView> createFromList(std::shared_ptr<ImageView>* listImages, size_t count);

  /**
   * Creates a MultiImageView from an ImageView
   * ImageView (whose dimension is greater than 2) is unstacked according to given dimension
   * into a MultiImageView.
   * i.e. ImageView 3D (XYZ) === unstack according to Y ==> MultiImageView containing Y ImageViews (XZ)
   * @param imageView ImageView whose dimension is greater than 2
   * @param idxDimensionUnstack index of dimension used for unstacking
   * @return a MultiImageView containing as many sub-frames as the size of given dimension in original Imageview
   * @throw InvalidArgument if input image is null
   * @throw Error if given ImageView has invalid properties
   */
  static std::shared_ptr<MultiImageView> unstack(std::shared_ptr<ImageView> imageView, size_t idxDimensionUnstack);

  /**
   * Creates a MultiImageView from an ImageView
   * ImageView (whose dimension is greater than 2) is unstacked according to given dimension
   * into a MultiImageView.
   * i.e. ImageView 3D (XYZ) === unstack according to Y ==> MultiImageView containing Y ImageViews (XZ)
   * @param imageView ImageView whose dimension is greater than 2
   * @param dimension ImageDimension to unstack
   * @return a MultiImageView containing as many sub-frames as the size of given dimension in original Imageview
   * @throw InvalidArgument if input image is null
   * @throw Error if dimension is not contained into ImageView
   * @throw Error if given ImageView has invalid properties
   */
  static std::shared_ptr<MultiImageView> unstack(std::shared_ptr<ImageView> imageView, ImageDimension dimension);
  /**
   * Creates a MultiImageView from an ImageView, containing as many ImageViews as there are channels in original Image
   * (one by channel). i.e.  ImageView RGB => MultiImageView of 3 ImageViews (ImageView RED, ImageView GREEN, ImageView
   * BLUE)
   * @param imageView ImageView which must be deinterlaced
   * @return a MultiImageView with one ImageView for each channel of original ImageView
   * @throw InvalidArgument if input image is null
   * @throw Error if given ImageView has invalid properties
   */
  static std::shared_ptr<MultiImageView> deinterlace(std::shared_ptr<ImageView> imageView);
};

} // end namespace iolink
