// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <memory>

#include <iolink/ArrayX.h>
#include <iolink/IOLinkAPI.h>
#include <iolink/RegionX.h>
#include <iolink/VectorX.h>
#include <iolink/view/Extension.h>

namespace iolink
{

class ImageView;

/**
 * ImageView extension to describe how an image is tiled.
 *
 * The ImageView is tiled with tiles corresponding to the size returned
 * with the method maxTileSize, so the tile can be smaller, when on the limits
 * of the view, they can be cropped for example.
 *
 * Tiles are indexed from 0 to tileCount() - 1, all indices values out of that
 * range are invalid. It is done so that you can iterate over these indices,
 * so you can process all tiles of the dataset.
 */
class IOLINK_API TilingInfoExtension : public Extension
{
public:
  /**
   * Identifier used to load the extension using the View::extension method.
   */
  static constexpr size_t EXTENSION_ID = 0x1;

  /**
   * @brief Upcast a generic extension to this specific type.
   *
   * @param extension The extension to cast.
   * @return The upcasted extension. This will be null if no upcasting is possible.
   */
  inline static std::shared_ptr<TilingInfoExtension> cast(const std::shared_ptr<Extension>& extension)
  {
    return std::dynamic_pointer_cast<TilingInfoExtension>(extension);
  }

  /**
   * Tile index for invalid tiles, used in the tiling grid.
   */
  static constexpr size_t INVALID_TILE_INDEX = 0xFFFFFFFF;

  /**
   * Return the maximum tile size used for the image.
   */
  virtual const VectorXu64& maxTileSize() const = 0;

  /**
   * Get the number of tiles in the image.
   */
  virtual size_t tileCount() const = 0;

  /**
   * Return the region of the chosen tile.
   *
   * @param tileIndex the index of the tile.
   * @throw Error If tileIndex is out of range.
   */
  virtual RegionXu64 tileRegion(size_t tileIndex) const = 0;

  /**
   * Return A N-dimensional array, with each element being a tile index.
   *
   * Some tiles can be invalid, because of holes for example. Their index
   * will be INVALID_TILE_INDEX.
   */
  virtual const ArrayXu64& tilingGrid() const = 0;

  /**
   * Read data from a tile into a memory buffer.
   *
   * In order to get the size of the destination buffer, you must use the
   * tileRegion method, and multiply the element count of that region by
   * the byte count of the image's data type.
   *
   * @param tileIndex The index of the tile to read.
   * @param dst The buffer where read data will be put.
   *
   * @throw Error When tileIndex is out of range.
   */
  virtual void readTile(size_t tileIndex, void* dst);

  /**
   * Write data from a memory buffer into a tile.
   *
   * In order to get the size of the source buffer, you must use the
   * tileRegion method, and multiply the element count of that region by
   * the byte count of the image's data type.
   *
   * @param tileIndex The index of the tile to write.
   * @param src The buffer that contains data to write.
   *
   * @throw Error When tileIndex is out of range.
   */
  virtual void writeTile(size_t tileIndex, const void* src);

protected:
  /**
   * Get the ImageView that is tiled by this instance.
   *
   * Used by readTile and writeTile default implementation.
   */
  virtual ImageView* image() const = 0;
};

} // end namespace iolink
