// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once
#include <memory>
#include <string>

#include <iolink/view/Extension.h>

namespace iolink
{
// forward declaration
class View;

/**
 *  Enum which represents the different locations where view data may come from.
 */
enum class ViewDataOrigin
{
  /**
   * The origin of data cannot be determined
   */
  UNKNOWN = 0,

  /**
   * Data is located in CPU memory.
   */
  CPU_MEMORY = 1,

  /**
   * Data is located on a drive (HDD, SSD, CD, etc.).
   */
  DRIVE = 2,

  /**
   * Data is located in GPU memory.
   */
  GPU_MEMORY = 3,

  /**
   * Data are retrieved through the network.
   */
  NETWORK = 4,

  /**
   * Data is generated and never stored.
   */
  GENERATED = 5,

  /**
   * Data come from many sources.
   */
  COMPOSITE = 6
};

/**
 * Extension used to access view's origins if it exists.
 * Views can be created from scratch, but also
 * from one or many other views.
 * This extension provides methods which allow to build the dependency tree
 * of current view. It also helps to retrieve information to know data origin and
 * to identify a view between others.
 */
class IOLINK_API ViewOriginExtension : public Extension
{
public:
  /**
   * Identifier used to load the extension using the View::extension method.
   */
  static constexpr size_t EXTENSION_ID = 0x3;

  /**
   * @brief Upcast a generic extension to this specific type.
   *
   * @param extension The extension to cast.
   * @return The upcasted extension. This will be null if no upcasting is possible.
   */
  inline static std::shared_ptr<ViewOriginExtension> cast(const std::shared_ptr<Extension>& extension)
  {
    return std::dynamic_pointer_cast<ViewOriginExtension>(extension);
  }

  /**
   * Returns an unique identifier for current view
   *
   * @return view identifier
   */
  virtual uint64_t uid() const = 0;

  /**
   * Returns the count of parents used to create
   * the current view.
   *
   * @return count of view's parents
   */
  virtual size_t parentCount() const = 0;

  /**
   * Returns, for the given index, the viewOrigin extension of the parent used
   * to create the current view.
   *
   * @param idx index of parent to retrieve
   * @throw Error if the index is out of range
   * @return a ViewOrigin extension which cannot be null
   */
  virtual std::shared_ptr<ViewOriginExtension> parent(size_t idx) const = 0;

  /**
   * Returns the data origin of current view
   *
   * @throw Error if data origin is not available
   */
  virtual ViewDataOrigin dataOrigin() const = 0;

  /**
   * Returns the name of the current view to display
   *
   * @throw Error if display name is not available
   */
  virtual std::string displayName() const = 0;

  /**
   * String representation of the extension.
   */
  std::string toString() const;
};

} // end namespace iolink
