/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Data/Kernels/MorphologicalKernel3d.h>
#include <ImageDev/ImageDevCppExports.h>
#include <memory>

namespace imagedev {

class MorphologicalLut3dPrivate;

/**
 * This class represents a 3D morphological lut.
 */
class IMAGEDEV_CPP_API MorphologicalLut3d
{
public:
    typedef MorphologicalLut3d Self;
    typedef std::shared_ptr< Self > Ptr;

    // constructor returning a shared ptr on an empty MorphologicalLut3d
    static Ptr New();

    // constructor returning a shared ptr on a MorphologicalLut3d with a kernel with a specific rotation
    static Ptr New( const MorphologicalKernel3d::Ptr kernel,
                    const size_t rotationX,
                    const size_t rotationY,
                    const size_t rotationZ );

    // constructor returning a shared ptr on a MorphologicalLut3d with a kernel in all possible rotations
    static Ptr New( const MorphologicalKernel3d::Ptr kernel );

    /// destructor
    ~MorphologicalLut3d();

    /**
     * Add a kernel to the LUT
     * @param[in] kernel The kernel to add into the LUT.
     * A 3D-kernel should contain exactly 27 values and indexes matching the following table.
     * 0 1 2            9 10 11           18 19 20
     * 3 4 5 for z=0,  12 13 14 for z=1,  21 22 23  for z=2
     * 6 7 8           15 16 17           24 25 26
     * Accepted values for the kernel are MorphologicalKernel3d::Value:
     * - ZERO: pixel is off
     * - ONE: pixel is on
     * - ANY: pixel is ignored
     * @param[in] rotationX number of rotation on X axis of 2 PI / 8 of the kernel before insertion.
     * @param[in] rotationY number of rotation on Y axis of 2 PI / 8 of the kernel before insertion.
     * @param[in] rotationY number of rotation on Z axis of 2 PI / 8 of the kernel before insertion.
     */
    void addKernel( const MorphologicalKernel3d::Ptr kernel,
                    const size_t rotationX = 0,
                    const size_t rotationY = 0,
                    const size_t rotationZ = 0 );

    /**
     * Add a kernel to the LUT with all possible rotations
     * @param[in] kernel The kernel to add into the LUT.
     * A 3D-kernel should contain exactly 27 values and indexes matching the following table.
     * 0 1 2            9 10 11           18 19 20
     * 3 4 5 for z=0,  12 13 14 for z=1,  21 22 23  for z=2
     * 6 7 8           15 16 17           24 25 26
     * Accepted values for the kernel are MorphologicalKernel3d::Value:
     * - ZERO: pixel is off
     * - ONE: pixel is on
     * - ANY: pixel is ignored
     */
    void addKernels( const MorphologicalKernel3d::Ptr kernel );

    /**
     * Reset the LUT to initial values (empty).
     */
    void reset();

    /**
     * Return the number of kernels in the lut.
     */
    size_t kernelCount();

    friend class MorphologicalLut3dPrivate;

protected:
    MorphologicalLut3d();
    MorphologicalLut3dPrivate* m_p;
};

} // namespace imagedev
