/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>
#include <iolink/Matrix.h>

namespace imagedev
{
/// Computes the best transformation for the co-registration of two images, using an iterative optimization algorithm.
class IMAGEDEV_CPP_API AffineRegistration final : public GenericAlgorithm
{
public:
    /// The way to define the intensity range to be considered by the algorithm in the moving image.
    enum RangeModeFixed
    {
    /// The intensity range of the fixed image is between the minimum and the maximum of the image intensities.
        R_MIN_MAX = 0,
    /// The intensity range of the fixed image is between user-defined bounds [a,b].
        R_OTHER
    };
    /// The way to define the intensity range to be considered by the algorithm in the fixed image.
    enum RangeModeMoving
    {
    /// The intensity range of the moving image is between the minimum and the maximum of the image intensities.
        M_MIN_MAX = 0,
    /// The intensity range of the moving image is between user-defined bounds [a,b].
        M_OTHER
    };
    /// Selects the type of transformation to estimate. It determines the number of degrees of freedom of the transformation.
    enum TransformType
    {
    /// Translation only.
        TRANSLATION = 0,
    /// Rigid transformation consisting of translation and rotation.
        RIGID,
    /// Transformation consisting of translation, rotation, and scale (only one scale for all dimensions).
        RIGID_ISOTROPIC_SCALING,
    /// Transformation consisting of translation, rotation, and scales (one scale per dimension).
        RIGID_ANISOTROPIC_SCALING,
    /// Affine transformation consisting of translation, rotation, scale, and shear.
        AFFINE
    };
    /// The metric type to compute.
    enum MetricType
    {
    /// The mean squared difference between the gray values of the moving and fixed image.
        EUCLIDEAN = 0,
    /// The correlation ratio of the registered images.
        CORRELATION,
    /// The information shared by both images.
        NORMALIZED_MUTUAL_INFORMATION
    };
    /// The optimizer strategy.It determines the descent gradient method that is applied to iteratively modify the transformation to produce a better similarity measure.
    enum OptimizerType
    {
    /// The Extensive direction strategy. It is similar to the best neighbor by integrating a gradient factor concept.
        EXTENSIVE_DIRECTION = 0,
    /// This Best neighbour strategy. It is the most straighforward.
        BEST_NEIGHBOUR,
    /// The quasi-Newton strategy. The descent gradient used to find the best transformation is based on the quasi-Newton method.
        QUASI_NEWTON,
    /// The conjugated gradients strategy. The descent gradient used to find the best transformation is based on the conjugated gradients method.
        CONJUGATED_GRADIENTS,
    /// The line search strategy. It is roughly the same as the extensive direction except that it keeps incrementing the transformation as long as the metric is improved.
        LINE_SEARCH
    };

    // Command constructor.
    AffineRegistration();


    /// Gets the inputMovingImage parameter.
    /// The input moving image, also known as the model image.
    std::shared_ptr< iolink::ImageView > inputMovingImage() const;
    /// Sets the inputMovingImage parameter.
    /// The input moving image, also known as the model image.
    void setInputMovingImage( std::shared_ptr< iolink::ImageView > inputMovingImage );

    /// Gets the inputFixedImage parameter.
    /// The input fixed image, also known as the reference image.
    std::shared_ptr< iolink::ImageView > inputFixedImage() const;
    /// Sets the inputFixedImage parameter.
    /// The input fixed image, also known as the reference image.
    void setInputFixedImage( std::shared_ptr< iolink::ImageView > inputFixedImage );

    /// Gets the initialTransform parameter.
    /// The initial transformation, which pre-aligns the moving image onto the fixed image. It is represented by a 4x4 matrix.
    iolink::Matrix4d initialTransform() const;
    /// Sets the initialTransform parameter.
    /// The initial transformation, which pre-aligns the moving image onto the fixed image. It is represented by a 4x4 matrix.
    void setInitialTransform( const iolink::Matrix4d& initialTransform );

    /// Gets the autoParameterMode parameter.
    /// Determines whether or not the coarsest resampling factor and optimizer steps are automatically computed.
    bool autoParameterMode() const;
    /// Sets the autoParameterMode parameter.
    /// Determines whether or not the coarsest resampling factor and optimizer steps are automatically computed.
    void setAutoParameterMode( const bool& autoParameterMode );

    /// Gets the optimizerStep parameter.
    /// The step sizes, in world coordinates, used by the optimizer at coarsest and finest scales.
    iolink::Vector2d optimizerStep() const;
    /// Sets the optimizerStep parameter.
    /// The step sizes, in world coordinates, used by the optimizer at coarsest and finest scales.
    void setOptimizerStep( const iolink::Vector2d& optimizerStep );

    /// Gets the intensityRangeFixed parameter.
    /// The range [a,b] of gray values for the fixed data set.
    iolink::Vector2d intensityRangeFixed() const;
    /// Sets the intensityRangeFixed parameter.
    /// The range [a,b] of gray values for the fixed data set.
    void setIntensityRangeFixed( const iolink::Vector2d& intensityRangeFixed );

    /// Gets the intensityRangeMoving parameter.
    /// The range [a,b] of gray values for the moving data set.
    iolink::Vector2d intensityRangeMoving() const;
    /// Sets the intensityRangeMoving parameter.
    /// The range [a,b] of gray values for the moving data set.
    void setIntensityRangeMoving( const iolink::Vector2d& intensityRangeMoving );

    /// Gets the coarsestResampling parameter.
    /// The sub-sampling factor along each axis.
    iolink::Vector3i32 coarsestResampling() const;
    /// Sets the coarsestResampling parameter.
    /// The sub-sampling factor along each axis.
    void setCoarsestResampling( const iolink::Vector3i32& coarsestResampling );

    /// Gets the rangeModeFixed parameter.
    /// The way to define the intensity range to be considered by the algorithm in the moving image.
    AffineRegistration::RangeModeFixed rangeModeFixed() const;
    /// Sets the rangeModeFixed parameter.
    /// The way to define the intensity range to be considered by the algorithm in the moving image.
    void setRangeModeFixed( const AffineRegistration::RangeModeFixed& rangeModeFixed );

    /// Gets the rangeModeMoving parameter.
    /// The way to define the intensity range to be considered by the algorithm in the fixed image.
    AffineRegistration::RangeModeMoving rangeModeMoving() const;
    /// Sets the rangeModeMoving parameter.
    /// The way to define the intensity range to be considered by the algorithm in the fixed image.
    void setRangeModeMoving( const AffineRegistration::RangeModeMoving& rangeModeMoving );

    /// Gets the transformType parameter.
    /// Selects the type of transformation to estimate. It determines the number of degrees of freedom of the transformation.
    AffineRegistration::TransformType transformType() const;
    /// Sets the transformType parameter.
    /// Selects the type of transformation to estimate. It determines the number of degrees of freedom of the transformation.
    void setTransformType( const AffineRegistration::TransformType& transformType );

    /// Gets the ignoreFinestLevel parameter.
    /// Skip the finest level of the pyramid if equal to true.
    bool ignoreFinestLevel() const;
    /// Sets the ignoreFinestLevel parameter.
    /// Skip the finest level of the pyramid if equal to true.
    void setIgnoreFinestLevel( const bool& ignoreFinestLevel );

    /// Gets the metricType parameter.
    /// The metric type to compute.
    AffineRegistration::MetricType metricType() const;
    /// Sets the metricType parameter.
    /// The metric type to compute.
    void setMetricType( const AffineRegistration::MetricType& metricType );

    /// Gets the thresholdOutside parameter.
    /// The factor used to define the importance of images coverage.
    double thresholdOutside() const;
    /// Sets the thresholdOutside parameter.
    /// The factor used to define the importance of images coverage.
    void setThresholdOutside( const double& thresholdOutside );

    /// Gets the optimizerType parameter.
    /// The optimizer strategy.It determines the descent gradient method that is applied to iteratively modify the transformation to produce a better similarity measure.
    AffineRegistration::OptimizerType optimizerType() const;
    /// Sets the optimizerType parameter.
    /// The optimizer strategy.It determines the descent gradient method that is applied to iteratively modify the transformation to produce a better similarity measure.
    void setOptimizerType( const AffineRegistration::OptimizerType& optimizerType );

    /// Gets the outputTransform parameter.
    /// The output transformation, which aligns the moving image onto the fixed image. It is represented by a 4x4 matrix.
    iolink::Matrix4d outputTransform() const;

    // Method to launch the command.
    void execute();

};

/// Computes the best transformation for the co-registration of two images, using an iterative optimization algorithm.
/// @param inputMovingImage The input moving image, also known as the model image.
/// @param inputFixedImage The input fixed image, also known as the reference image.
/// @param initialTransform The initial transformation, which pre-aligns the moving image onto the fixed image. It is represented by a 4x4 matrix.
/// @param autoParameterMode Determines whether or not the coarsest resampling factor and optimizer steps are automatically computed.
/// @param optimizerStep The step sizes, in world coordinates, used by the optimizer at coarsest and finest scales.
/// @param intensityRangeFixed The range [a,b] of gray values for the fixed data set.
/// @param intensityRangeMoving The range [a,b] of gray values for the moving data set.
/// @param coarsestResampling The sub-sampling factor along each axis.
/// @param rangeModeFixed The way to define the intensity range to be considered by the algorithm in the moving image.
/// @param rangeModeMoving The way to define the intensity range to be considered by the algorithm in the fixed image.
/// @param transformType Selects the type of transformation to estimate. It determines the number of degrees of freedom of the transformation.
/// @param ignoreFinestLevel Skip the finest level of the pyramid if equal to true.
/// @param metricType The metric type to compute.
/// @param thresholdOutside The factor used to define the importance of images coverage.
/// @param optimizerType The optimizer strategy.It determines the descent gradient method that is applied to iteratively modify the transformation to produce a better similarity measure.
/// @param outputTransform The output transformation, which aligns the moving image onto the fixed image. It is represented by a 4x4 matrix.
/// @return Returns the outputTransform output parameter.
IMAGEDEV_CPP_API 
iolink::Matrix4d
affineRegistration( std::shared_ptr< iolink::ImageView > inputMovingImage,
                    std::shared_ptr< iolink::ImageView > inputFixedImage,
                    const iolink::Matrix4d& initialTransform,
                    bool autoParameterMode,
                    const iolink::Vector2d& optimizerStep,
                    const iolink::Vector2d& intensityRangeFixed,
                    const iolink::Vector2d& intensityRangeMoving,
                    const iolink::Vector3i32& coarsestResampling,
                    AffineRegistration::RangeModeFixed rangeModeFixed,
                    AffineRegistration::RangeModeMoving rangeModeMoving,
                    AffineRegistration::TransformType transformType,
                    bool ignoreFinestLevel,
                    AffineRegistration::MetricType metricType,
                    double thresholdOutside,
                    AffineRegistration::OptimizerType optimizerType );
} // namespace imagedev
