/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <ImageDev/Data/Kernels/MorphologicalLut3d.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/GlobalMeasurements/ApplyMorphologicalLutMsr.h>

namespace imagedev
{
// Output structure of the applyMorphologicalLut3d function.
struct ApplyMorphologicalLut3dOutput
{
    /// The binary output image. Its size and type are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputBinaryImage;
    /// The output object containing the number of modified voxels by the selected LUT.
    ApplyMorphologicalLutMsr::Ptr outputMeasurement;
};

/// Applies a predefined morphological lookup table (LUT) on a three-dimensional binary image.
class IMAGEDEV_CPP_API ApplyMorphologicalLut3d final : public GenericAlgorithm
{
public:
    /// The value to set when a predefined configuration is encountered.
    enum MatchingFlag
    {
    /// The output value is forced to 0.
        M_SET_ZERO = 0,
    /// The output value is forced to 1.
        M_SET_ONE,
    /// The output value is the logical negation of the input value.
        M_CHANGE,
    /// The output value is equal to the input value.
        M_PRESERVE
    };
    /// The value to set when a predefined configuration is not encountered.
    enum UnmatchingFlag
    {
    /// The output value is forced to 0.
        U_SET_ZERO = 0,
    /// The output value is forced to 1.
        U_SET_ONE,
    /// The output value is the logical negation of the input value.
        U_CHANGE,
    /// The output value is equal to the input value.
        U_PRESERVE
    };
    /// The border condition used to extrapolate outside pixels.
    enum BorderCondition
    {
    /// Voxels outside of the image are considered as having the value 0.
        ZERO = 0,
    /// Voxels outside of the image are a replication the border image pixels.
        MIRROR
    };

    // Command constructor.
    ApplyMorphologicalLut3d();


    /// Gets the inputBinaryImage parameter.
    /// The binary input image.
    std::shared_ptr< iolink::ImageView > inputBinaryImage() const;
    /// Sets the inputBinaryImage parameter.
    /// The binary input image.
    void setInputBinaryImage( std::shared_ptr< iolink::ImageView > inputBinaryImage );

    /// Gets the morphologicalLut parameter.
    /// The input morphological LUT array defining the configurations to detect.
    imagedev::MorphologicalLut3d::Ptr morphologicalLut() const;
    /// Sets the morphologicalLut parameter.
    /// The input morphological LUT array defining the configurations to detect.
    void setMorphologicalLut( const imagedev::MorphologicalLut3d::Ptr& morphologicalLut );

    /// Gets the matchingFlag parameter.
    /// The value to set when a predefined configuration is encountered.
    ApplyMorphologicalLut3d::MatchingFlag matchingFlag() const;
    /// Sets the matchingFlag parameter.
    /// The value to set when a predefined configuration is encountered.
    void setMatchingFlag( const ApplyMorphologicalLut3d::MatchingFlag& matchingFlag );

    /// Gets the unmatchingFlag parameter.
    /// The value to set when a predefined configuration is not encountered.
    ApplyMorphologicalLut3d::UnmatchingFlag unmatchingFlag() const;
    /// Sets the unmatchingFlag parameter.
    /// The value to set when a predefined configuration is not encountered.
    void setUnmatchingFlag( const ApplyMorphologicalLut3d::UnmatchingFlag& unmatchingFlag );

    /// Gets the borderCondition parameter.
    /// The border condition used to extrapolate outside pixels.
    ApplyMorphologicalLut3d::BorderCondition borderCondition() const;
    /// Sets the borderCondition parameter.
    /// The border condition used to extrapolate outside pixels.
    void setBorderCondition( const ApplyMorphologicalLut3d::BorderCondition& borderCondition );

    /// Gets the outputBinaryImage parameter.
    /// The binary output image. Its size and type are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputBinaryImage() const;
    /// Sets the outputBinaryImage parameter.
    /// The binary output image. Its size and type are forced to the same values as the input.
    void setOutputBinaryImage( std::shared_ptr< iolink::ImageView > outputBinaryImage );

    /// Gets the outputMeasurement parameter.
    /// The output object containing the number of modified voxels by the selected LUT.
    ApplyMorphologicalLutMsr::Ptr outputMeasurement() const;

    // Method to launch the command.
    void execute();

};

/// Applies a predefined morphological lookup table (LUT) on a three-dimensional binary image.
/// @param inputBinaryImage The binary input image.
/// @param morphologicalLut The input morphological LUT array defining the configurations to detect.
/// @param matchingFlag The value to set when a predefined configuration is encountered.
/// @param unmatchingFlag The value to set when a predefined configuration is not encountered.
/// @param borderCondition The border condition used to extrapolate outside pixels.
/// @param outputBinaryImage The binary output image. Its size and type are forced to the same values as the input.
/// @param outputMeasurement The output object containing the number of modified voxels by the selected LUT.
/// @return Returns a ApplyMorphologicalLut3dOutput structure containing the outputBinaryImage and outputMeasurement output parameters.
IMAGEDEV_CPP_API 
ApplyMorphologicalLut3dOutput
applyMorphologicalLut3d( std::shared_ptr< iolink::ImageView > inputBinaryImage,
                         imagedev::MorphologicalLut3d::Ptr morphologicalLut,
                         ApplyMorphologicalLut3d::MatchingFlag matchingFlag,
                         ApplyMorphologicalLut3d::UnmatchingFlag unmatchingFlag,
                         ApplyMorphologicalLut3d::BorderCondition borderCondition,
                         std::shared_ptr< iolink::ImageView > outputBinaryImage = nullptr,
                         ApplyMorphologicalLutMsr::Ptr outputMeasurement = nullptr );
} // namespace imagedev
