/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/GlobalMeasurements/AxisPropagationLabel3dMsr.h>

namespace imagedev
{
// Output structure of the axisPropagationLabel3d function.
struct AxisPropagationLabel3dOutput
{
    /// The binary output image representing the propagation within the selected phase. A void image means that there is no connection through the selected axis. Its dimensions are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputLabelImage;
    /// The output measurement result.
    AxisPropagationLabel3dMsr::Ptr outputMeasurement;
};

/// Simulates the propagation through a phase of a three-dimensional image and computes a label image representing separately the connected and unconnected voxels of this phase.
class IMAGEDEV_CPP_API AxisPropagationLabel3d final : public GenericAlgorithm
{
public:
    /// The axis along which the propagation is performed.
    enum Axis
    {
    /// The propagation is performed along the X axis.
        X_AXIS = 0,
    /// The propagation is performed along the Y axis.
        Y_AXIS,
    /// The propagation is performed along the Z axis.
        Z_AXIS
    };

    // Command constructor.
    AxisPropagationLabel3d();


    /// Gets the inputImage parameter.
    /// The input grayscale, binary, or color image. In the case of a binary image, the background represents the phase where the propagation has to be performed.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input grayscale, binary, or color image. In the case of a binary image, the background represents the phase where the propagation has to be performed.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the maskAndSeedValues parameter.
    /// The first value is a mask value. Voxels with this intensity value are not infiltrated by the propagation. The second value is the seed value. Voxels with an intensity value lower than this seed value are not infiltrated by the propagation. A value lower than the second value of maskAndSeedValues or greater or equal to maximumThreshold is ignored. This parameter is ignored in binary case, and is forced to [2, 0].
    iolink::Vector2i32 maskAndSeedValues() const;
    /// Sets the maskAndSeedValues parameter.
    /// The first value is a mask value. Voxels with this intensity value are not infiltrated by the propagation. The second value is the seed value. Voxels with an intensity value lower than this seed value are not infiltrated by the propagation. A value lower than the second value of maskAndSeedValues or greater or equal to maximumThreshold is ignored. This parameter is ignored in binary case, and is forced to [2, 0].
    void setMaskAndSeedValues( const iolink::Vector2i32& maskAndSeedValues );

    /// Gets the maximumThreshold parameter.
    /// The maximum propagation intensity value. Voxels having an intensity value greater or equal to this value are rejected by the propagation. This parameter is ignored in binary case, and is forced to 1.
    int32_t maximumThreshold() const;
    /// Sets the maximumThreshold parameter.
    /// The maximum propagation intensity value. Voxels having an intensity value greater or equal to this value are rejected by the propagation. This parameter is ignored in binary case, and is forced to 1.
    void setMaximumThreshold( const int32_t& maximumThreshold );

    /// Gets the axis parameter.
    /// The axis along which the propagation is performed.
    AxisPropagationLabel3d::Axis axis() const;
    /// Sets the axis parameter.
    /// The axis along which the propagation is performed.
    void setAxis( const AxisPropagationLabel3d::Axis& axis );

    /// Gets the outputLabelImage parameter.
    /// The binary output image representing the propagation within the selected phase. A void image means that there is no connection through the selected axis. Its dimensions are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputLabelImage() const;
    /// Sets the outputLabelImage parameter.
    /// The binary output image representing the propagation within the selected phase. A void image means that there is no connection through the selected axis. Its dimensions are forced to the same values as the input.
    void setOutputLabelImage( std::shared_ptr< iolink::ImageView > outputLabelImage );

    /// Gets the outputMeasurement parameter.
    /// The output measurement result.
    AxisPropagationLabel3dMsr::Ptr outputMeasurement() const;

    // Method to launch the command.
    void execute();

};

/// Simulates the propagation through a phase of a three-dimensional image and computes a label image representing separately the connected and unconnected voxels of this phase.
/// @param inputImage The input grayscale, binary, or color image. In the case of a binary image, the background represents the phase where the propagation has to be performed.
/// @param maskAndSeedValues The first value is a mask value. Voxels with this intensity value are not infiltrated by the propagation. The second value is the seed value. Voxels with an intensity value lower than this seed value are not infiltrated by the propagation. A value lower than the second value of maskAndSeedValues or greater or equal to maximumThreshold is ignored. This parameter is ignored in binary case, and is forced to [2, 0].
/// @param maximumThreshold The maximum propagation intensity value. Voxels having an intensity value greater or equal to this value are rejected by the propagation. This parameter is ignored in binary case, and is forced to 1.
/// @param axis The axis along which the propagation is performed.
/// @param outputLabelImage The binary output image representing the propagation within the selected phase. A void image means that there is no connection through the selected axis. Its dimensions are forced to the same values as the input.
/// @param outputMeasurement The output measurement result.
/// @return Returns a AxisPropagationLabel3dOutput structure containing the outputLabelImage and outputMeasurement output parameters.
IMAGEDEV_CPP_API 
AxisPropagationLabel3dOutput
axisPropagationLabel3d( std::shared_ptr< iolink::ImageView > inputImage,
                        const iolink::Vector2i32& maskAndSeedValues,
                        int32_t maximumThreshold,
                        AxisPropagationLabel3d::Axis axis,
                        std::shared_ptr< iolink::ImageView > outputLabelImage = nullptr,
                        AxisPropagationLabel3dMsr::Ptr outputMeasurement = nullptr );
} // namespace imagedev
