/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a beam hardening correction dedicated to cylindrical objects of a Computed Tomography volume.
class IMAGEDEV_CPP_API BeamHardeningCorrection3d final : public GenericAlgorithm
{
public:
    /// The axis along which the correction has to be carried out.
    enum AxisDirection
    {
    /// The correction is applied on YZ slices.
        X_AXIS = 0,
    /// The correction is applied on XZ slices.
        Y_AXIS,
    /// The correction is applied on XY slices.
        Z_AXIS
    };
    /// Set the cylinder axis to be used during correction: straight or sheared. This option is ignored if the axis direction is not Z_AXIS (forced to straight).
    enum AxisOrientation
    {
    /// A straight axis assumes that the sample is a vertical cylinder.
        STRAIGHT = 0,
    /// A sheared axis considers a sheared cylinder determined by the midpoints at the top and bottom of the sample.@BR This option is only available when the rotation axis is Z_AXIS at this stage.
        SHEARED
    };
    /// The way to define the main center of the cylindrical sample. It is ignored if the axis orientation is set to SHEARED (forced to OTHER).
    enum CenterMode
    {
    /// The correction center is the center of the input image.
        IMAGE_CENTER = 0,
    /// The correction center is user-defined.
        OTHER
    };
    /// The way to manage uncomputed values (that is, voxels that are out of the maximum radius).
    enum PaddingPolicy
    {
    /// The input intensity value is transferred.
        INPUT_VALUE = 0,
    /// The uncomputed area is filled with a padding value.
        FIXED_VALUE
    };

    // Command constructor.
    BeamHardeningCorrection3d();


    /// Gets the inputImage parameter.
    /// The 3D input image to correct.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The 3D input image to correct.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the axisDirection parameter.
    /// The axis along which the correction has to be carried out.
    BeamHardeningCorrection3d::AxisDirection axisDirection() const;
    /// Sets the axisDirection parameter.
    /// The axis along which the correction has to be carried out.
    void setAxisDirection( const BeamHardeningCorrection3d::AxisDirection& axisDirection );

    /// Gets the axisOrientation parameter.
    /// Set the cylinder axis to be used during correction: straight or sheared. This option is ignored if the axis direction is not Z_AXIS (forced to straight).
    BeamHardeningCorrection3d::AxisOrientation axisOrientation() const;
    /// Sets the axisOrientation parameter.
    /// Set the cylinder axis to be used during correction: straight or sheared. This option is ignored if the axis direction is not Z_AXIS (forced to straight).
    void setAxisOrientation( const BeamHardeningCorrection3d::AxisOrientation& axisOrientation );

    /// Gets the centerMode parameter.
    /// The way to define the main center of the cylindrical sample. It is ignored if the axis orientation is set to SHEARED (forced to OTHER).
    BeamHardeningCorrection3d::CenterMode centerMode() const;
    /// Sets the centerMode parameter.
    /// The way to define the main center of the cylindrical sample. It is ignored if the axis orientation is set to SHEARED (forced to OTHER).
    void setCenterMode( const BeamHardeningCorrection3d::CenterMode& centerMode );

    /// Gets the mainCenter parameter.
    /// The midpoint of the sample in voxel coordinates.
    iolink::Vector3u32 mainCenter() const;
    /// Sets the mainCenter parameter.
    /// The midpoint of the sample in voxel coordinates.
    void setMainCenter( const iolink::Vector3u32& mainCenter );

    /// Gets the secondaryCenter parameter.
    /// The midpoint of the last plane of the sample in voxel coordinates (ignored with STRAIGHT axis setting).
    iolink::Vector3u32 secondaryCenter() const;
    /// Sets the secondaryCenter parameter.
    /// The midpoint of the last plane of the sample in voxel coordinates (ignored with STRAIGHT axis setting).
    void setSecondaryCenter( const iolink::Vector3u32& secondaryCenter );

    /// Gets the radiusPercentage parameter.
    /// The portion, in percent, of the maximum radius that is effectively considered for computing the normalization factor. The average value of the area inside this percentage of the maximum radius is used to normalize the data.
    double radiusPercentage() const;
    /// Sets the radiusPercentage parameter.
    /// The portion, in percent, of the maximum radius that is effectively considered for computing the normalization factor. The average value of the area inside this percentage of the maximum radius is used to normalize the data.
    void setRadiusPercentage( const double& radiusPercentage );

    /// Gets the radiusRange parameter.
    /// The minimum and maximum radius, in voxel units, between which the correction is done. If the maximum radius is equal to 0, the largest possible radius is used.
    iolink::Vector2u32 radiusRange() const;
    /// Sets the radiusRange parameter.
    /// The minimum and maximum radius, in voxel units, between which the correction is done. If the maximum radius is equal to 0, the largest possible radius is used.
    void setRadiusRange( const iolink::Vector2u32& radiusRange );

    /// Gets the thresholdRange parameter.
    /// Values out of this range are not included in the statistics to calculate the correction.
    iolink::Vector2d thresholdRange() const;
    /// Sets the thresholdRange parameter.
    /// Values out of this range are not included in the statistics to calculate the correction.
    void setThresholdRange( const iolink::Vector2d& thresholdRange );

    /// Gets the standardDeviation parameter.
    /// The standard deviation, in voxel units, of the gaussian filtering used to smooth the histogram before performing the correction. No smoothing is applied if this value is set to 0.
    double standardDeviation() const;
    /// Sets the standardDeviation parameter.
    /// The standard deviation, in voxel units, of the gaussian filtering used to smooth the histogram before performing the correction. No smoothing is applied if this value is set to 0.
    void setStandardDeviation( const double& standardDeviation );

    /// Gets the paddingPolicy parameter.
    /// The way to manage uncomputed values (that is, voxels that are out of the maximum radius).
    BeamHardeningCorrection3d::PaddingPolicy paddingPolicy() const;
    /// Sets the paddingPolicy parameter.
    /// The way to manage uncomputed values (that is, voxels that are out of the maximum radius).
    void setPaddingPolicy( const BeamHardeningCorrection3d::PaddingPolicy& paddingPolicy );

    /// Gets the paddingValue parameter.
    /// The intensity value used to fill uncomputed voxels (that is, voxels that are out of the maximum radius). It is used only with the fixed value padding policy.
    double paddingValue() const;
    /// Sets the paddingValue parameter.
    /// The intensity value used to fill uncomputed voxels (that is, voxels that are out of the maximum radius). It is used only with the fixed value padding policy.
    void setPaddingValue( const double& paddingValue );

    /// Gets the outputImage parameter.
    /// The corrected image. Its dimensions and type are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The corrected image. Its dimensions and type are forced to the same values as the input.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Performs a beam hardening correction dedicated to cylindrical objects of a Computed Tomography volume.
/// @param inputImage The 3D input image to correct.
/// @param axisDirection The axis along which the correction has to be carried out.
/// @param axisOrientation Set the cylinder axis to be used during correction: straight or sheared. This option is ignored if the axis direction is not Z_AXIS (forced to straight).
/// @param centerMode The way to define the main center of the cylindrical sample. It is ignored if the axis orientation is set to SHEARED (forced to OTHER).
/// @param mainCenter The midpoint of the sample in voxel coordinates.
/// @param secondaryCenter The midpoint of the last plane of the sample in voxel coordinates (ignored with STRAIGHT axis setting).
/// @param radiusPercentage The portion, in percent, of the maximum radius that is effectively considered for computing the normalization factor. The average value of the area inside this percentage of the maximum radius is used to normalize the data.
/// @param radiusRange The minimum and maximum radius, in voxel units, between which the correction is done. If the maximum radius is equal to 0, the largest possible radius is used.
/// @param thresholdRange Values out of this range are not included in the statistics to calculate the correction.
/// @param standardDeviation The standard deviation, in voxel units, of the gaussian filtering used to smooth the histogram before performing the correction. No smoothing is applied if this value is set to 0.
/// @param paddingPolicy The way to manage uncomputed values (that is, voxels that are out of the maximum radius).
/// @param paddingValue The intensity value used to fill uncomputed voxels (that is, voxels that are out of the maximum radius). It is used only with the fixed value padding policy.
/// @param outputImage The corrected image. Its dimensions and type are forced to the same values as the input.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
beamHardeningCorrection3d( std::shared_ptr< iolink::ImageView > inputImage,
                           BeamHardeningCorrection3d::AxisDirection axisDirection,
                           BeamHardeningCorrection3d::AxisOrientation axisOrientation,
                           BeamHardeningCorrection3d::CenterMode centerMode,
                           const iolink::Vector3u32& mainCenter,
                           const iolink::Vector3u32& secondaryCenter,
                           double radiusPercentage,
                           const iolink::Vector2u32& radiusRange,
                           const iolink::Vector2d& thresholdRange,
                           double standardDeviation,
                           BeamHardeningCorrection3d::PaddingPolicy paddingPolicy,
                           double paddingValue,
                           std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
