/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a two or three-dimensional convolution between an input image $I$ and another image $K$ used as a convolution kernel.
class IMAGEDEV_CPP_API ConvolutionWithImage final : public GenericAlgorithm
{
public:
    /// The automatic intensity scaling mode.
    enum AutoScale
    {
    /// The result is not normalized.
        NO = 0,
    /// The result is automatically normalized by the sum of the absolute values of the kernel elements.
        YES
    };
    /// The algorithm implementation used to perform the convolution.
    enum FilterMode
    {
    /// The computation is performed by a multiplication in the Fourier domain. Using this mode provides a constant computation time independent from the kernel image size. This mode is also associated with higher consumption of memory.
        FREQUENCY = 0,
    /// The computation is performed applying a classic sliding window. Using this mode can provide a faster computation time, for small kernel image size.
        SPATIAL
    };

    // Command constructor.
    ConvolutionWithImage();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputKernelImage parameter.
    /// The convolution kernel image. In case of a multispectral input image it must have either 1 channels, or the same number of channel.
    std::shared_ptr< iolink::ImageView > inputKernelImage() const;
    /// Sets the inputKernelImage parameter.
    /// The convolution kernel image. In case of a multispectral input image it must have either 1 channels, or the same number of channel.
    void setInputKernelImage( std::shared_ptr< iolink::ImageView > inputKernelImage );

    /// Gets the autoScale parameter.
    /// The automatic intensity scaling mode.
    ConvolutionWithImage::AutoScale autoScale() const;
    /// Sets the autoScale parameter.
    /// The automatic intensity scaling mode.
    void setAutoScale( const ConvolutionWithImage::AutoScale& autoScale );

    /// Gets the filterMode parameter.
    /// The algorithm implementation used to perform the convolution.
    ConvolutionWithImage::FilterMode filterMode() const;
    /// Sets the filterMode parameter.
    /// The algorithm implementation used to perform the convolution.
    void setFilterMode( const ConvolutionWithImage::FilterMode& filterMode );

    /// Gets the outputImage parameter.
    /// The output image. Its dimensions are forced to the same values as the input. Its data type is float 64-bits in case of a 64-bits input, float 32-bits in other cases.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its dimensions are forced to the same values as the input. Its data type is float 64-bits in case of a 64-bits input, float 32-bits in other cases.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Performs a two or three-dimensional convolution between an input image $I$ and another image $K$ used as a convolution kernel.
/// @param inputImage The input image.
/// @param inputKernelImage The convolution kernel image. In case of a multispectral input image it must have either 1 channels, or the same number of channel.
/// @param autoScale The automatic intensity scaling mode.
/// @param filterMode The algorithm implementation used to perform the convolution.
/// @param outputImage The output image. Its dimensions are forced to the same values as the input. Its data type is float 64-bits in case of a 64-bits input, float 32-bits in other cases.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
convolutionWithImage( std::shared_ptr< iolink::ImageView > inputImage,
                      std::shared_ptr< iolink::ImageView > inputKernelImage,
                      ConvolutionWithImage::AutoScale autoScale,
                      ConvolutionWithImage::FilterMode filterMode,
                      std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
