/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <ImageDev/Data/Cuda/CudaContext.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/GlobalMeasurements/CrossCorrelation2dMsr.h>

namespace imagedev
{
// Output structure of the cudaCrossCorrelation2d function.
struct CudaCrossCorrelation2dOutput
{
    /// The output correlation image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputImage;
    /// The correlation matching results.
    CrossCorrelation2dMsr::Ptr outputMeasurement;
};

/// Performs a correlation by convolution between a two-dimensional grayscale image and a grayscale kernel. The calculations are performed on the GPU.
/// Warning: This command is experimental, his signature may be modified between now and his final version.
class IMAGEDEV_CPP_API CudaCrossCorrelation2d final : public GenericAlgorithm
{
public:
    /// The normalization mode for correlation.
    enum CorrelationMode
    {
    /// Direct correlation (no normalization).
        DIRECT = 0,
    /// Mean normalized correlation (luminosity).
        MEAN
    };
    /// The way to manage the GPU memory.
    enum TilingMode
    {
    /// The entire input image is transferred to the GPU memory.
        NONE = 0,
    /// The input image is processed by tiles of a predefined size.
        USER_DEFINED
    };

    // Command constructor.
    CudaCrossCorrelation2d();


    /// Gets the inputImage parameter.
    /// The input grayscale image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input grayscale image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputKernelImage parameter.
    /// The correlation kernel.
    std::shared_ptr< iolink::ImageView > inputKernelImage() const;
    /// Sets the inputKernelImage parameter.
    /// The correlation kernel.
    void setInputKernelImage( std::shared_ptr< iolink::ImageView > inputKernelImage );

    /// Gets the correlationMode parameter.
    /// The normalization mode for correlation.
    CudaCrossCorrelation2d::CorrelationMode correlationMode() const;
    /// Sets the correlationMode parameter.
    /// The normalization mode for correlation.
    void setCorrelationMode( const CudaCrossCorrelation2d::CorrelationMode& correlationMode );

    /// Gets the tilingMode parameter.
    /// The way to manage the GPU memory.
    CudaCrossCorrelation2d::TilingMode tilingMode() const;
    /// Sets the tilingMode parameter.
    /// The way to manage the GPU memory.
    void setTilingMode( const CudaCrossCorrelation2d::TilingMode& tilingMode );

    /// Gets the tileSize parameter.
    /// The tile width and height in pixels. They must be greater than or equal to the correlation kernel width and height. This parameter is used only in USER_DEFINED tiling mode.
    iolink::Vector2u32 tileSize() const;
    /// Sets the tileSize parameter.
    /// The tile width and height in pixels. They must be greater than or equal to the correlation kernel width and height. This parameter is used only in USER_DEFINED tiling mode.
    void setTileSize( const iolink::Vector2u32& tileSize );

    /// Gets the cudaContext parameter.
    /// CUDA context information.
    CudaContext::Ptr cudaContext() const;
    /// Sets the cudaContext parameter.
    /// CUDA context information.
    void setCudaContext( const CudaContext::Ptr& cudaContext );

    /// Gets the outputImage parameter.
    /// The output correlation image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output correlation image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    /// Gets the outputMeasurement parameter.
    /// The correlation matching results.
    CrossCorrelation2dMsr::Ptr outputMeasurement() const;

    // Method to launch the command.
    void execute();

};

/// Performs a correlation by convolution between a two-dimensional grayscale image and a grayscale kernel. The calculations are performed on the GPU.
/// Warning: This command is experimental, his signature may be modified between now and his final version.
/// @param inputImage The input grayscale image.
/// @param inputKernelImage The correlation kernel.
/// @param correlationMode The normalization mode for correlation.
/// @param tilingMode The way to manage the GPU memory.
/// @param tileSize The tile width and height in pixels. They must be greater than or equal to the correlation kernel width and height. This parameter is used only in USER_DEFINED tiling mode.
/// @param cudaContext CUDA context information.
/// @param outputImage The output correlation image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
/// @param outputMeasurement The correlation matching results.
/// @return Returns a CudaCrossCorrelation2dOutput structure containing the outputImage and outputMeasurement output parameters.
IMAGEDEV_CPP_API 
CudaCrossCorrelation2dOutput
cudaCrossCorrelation2d( std::shared_ptr< iolink::ImageView > inputImage,
                        std::shared_ptr< iolink::ImageView > inputKernelImage,
                        CudaCrossCorrelation2d::CorrelationMode correlationMode,
                        CudaCrossCorrelation2d::TilingMode tilingMode,
                        const iolink::Vector2u32& tileSize,
                        CudaContext::Ptr cudaContext,
                        std::shared_ptr< iolink::ImageView > outputImage = nullptr,
                        CrossCorrelation2dMsr::Ptr outputMeasurement = nullptr );
} // namespace imagedev
