/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the eigenDecomposition3d function.
struct EigenDecomposition3dOutput
{
    /// The first eigenvector output image containing the largest eigenvalue.
    std::shared_ptr< iolink::ImageView > outputVectorImage1;
    /// The second eigenvector image containing the medium eigenvalue.
    std::shared_ptr< iolink::ImageView > outputVectorImage2;
    /// The third eigenvector image containing the smallest eigenvalue.
    std::shared_ptr< iolink::ImageView > outputVectorImage3;
    /// The eigenvalues output image. Each channel corresponds to an eigenvalue, from the largest to the smallest.
    std::shared_ptr< iolink::ImageView > outputEigenvaluesImage;
};

/// Performs the singular value decomposition (SVD) of a 3D tensor field image.
class IMAGEDEV_CPP_API EigenDecomposition3d final : public GenericAlgorithm
{
public:
    /// The output images to be computed. Several outputs can be generated by combining the associated enumerated values.
    enum OutputSelection
    {
    /// Enable the computation of the first eigenvector, corresponding to the largest eigenvalue.
        EIGEN_VECTOR_1 = 1,
    /// Enable the computation of the second eigenvector, corresponding to the medium eigenvalue.
        EIGEN_VECTOR_2 = 2,
    /// Enable the computation of the third eigenvector, corresponding to the smallest eigenvalue.
        EIGEN_VECTOR_3 = 4,
    /// Enable the computation of the eigenvalues.
        EIGEN_VALUES = 8
    };

    // Command constructor.
    EigenDecomposition3d();


    /// Gets the inputTensorImage parameter.
    /// The input image of which each voxel represents a 3x3 symmetric matrix. This image must have the float data type and contain 6 channels.
    std::shared_ptr< iolink::ImageView > inputTensorImage() const;
    /// Sets the inputTensorImage parameter.
    /// The input image of which each voxel represents a 3x3 symmetric matrix. This image must have the float data type and contain 6 channels.
    void setInputTensorImage( std::shared_ptr< iolink::ImageView > inputTensorImage );

    /// Gets the outputSelection parameter.
    /// The output images to be computed. Several outputs can be generated by combining the associated enumerated values.
    int32_t outputSelection() const;
    /// Sets the outputSelection parameter.
    /// The output images to be computed. Several outputs can be generated by combining the associated enumerated values.
    void setOutputSelection( const int32_t& outputSelection );

    /// Gets the outputVectorImage1 parameter.
    /// The first eigenvector output image containing the largest eigenvalue.
    std::shared_ptr< iolink::ImageView > outputVectorImage1() const;
    /// Sets the outputVectorImage1 parameter.
    /// The first eigenvector output image containing the largest eigenvalue.
    void setOutputVectorImage1( std::shared_ptr< iolink::ImageView > outputVectorImage1 );

    /// Gets the outputVectorImage2 parameter.
    /// The second eigenvector image containing the medium eigenvalue.
    std::shared_ptr< iolink::ImageView > outputVectorImage2() const;
    /// Sets the outputVectorImage2 parameter.
    /// The second eigenvector image containing the medium eigenvalue.
    void setOutputVectorImage2( std::shared_ptr< iolink::ImageView > outputVectorImage2 );

    /// Gets the outputVectorImage3 parameter.
    /// The third eigenvector image containing the smallest eigenvalue.
    std::shared_ptr< iolink::ImageView > outputVectorImage3() const;
    /// Sets the outputVectorImage3 parameter.
    /// The third eigenvector image containing the smallest eigenvalue.
    void setOutputVectorImage3( std::shared_ptr< iolink::ImageView > outputVectorImage3 );

    /// Gets the outputEigenvaluesImage parameter.
    /// The eigenvalues output image. Each channel corresponds to an eigenvalue, from the largest to the smallest.
    std::shared_ptr< iolink::ImageView > outputEigenvaluesImage() const;
    /// Sets the outputEigenvaluesImage parameter.
    /// The eigenvalues output image. Each channel corresponds to an eigenvalue, from the largest to the smallest.
    void setOutputEigenvaluesImage( std::shared_ptr< iolink::ImageView > outputEigenvaluesImage );

    // Method to launch the command.
    void execute();

};

/// Performs the singular value decomposition (SVD) of a 3D tensor field image.
/// @param inputTensorImage The input image of which each voxel represents a 3x3 symmetric matrix. This image must have the float data type and contain 6 channels.
/// @param outputSelection The output images to be computed. Several outputs can be generated by combining the associated enumerated values.
/// @param outputVectorImage1 The first eigenvector output image containing the largest eigenvalue.
/// @param outputVectorImage2 The second eigenvector image containing the medium eigenvalue.
/// @param outputVectorImage3 The third eigenvector image containing the smallest eigenvalue.
/// @param outputEigenvaluesImage The eigenvalues output image. Each channel corresponds to an eigenvalue, from the largest to the smallest.
/// @return Returns a EigenDecomposition3dOutput structure containing the outputVectorImage1, outputVectorImage2, outputVectorImage3 and outputEigenvaluesImage output parameters.
IMAGEDEV_CPP_API 
EigenDecomposition3dOutput
eigenDecomposition3d( std::shared_ptr< iolink::ImageView > inputTensorImage,
                      int32_t outputSelection,
                      std::shared_ptr< iolink::ImageView > outputVectorImage1 = nullptr,
                      std::shared_ptr< iolink::ImageView > outputVectorImage2 = nullptr,
                      std::shared_ptr< iolink::ImageView > outputVectorImage3 = nullptr,
                      std::shared_ptr< iolink::ImageView > outputEigenvaluesImage = nullptr );
} // namespace imagedev
