/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Applies a three-dimensional Gaussian filter using either a separable finite kernel, or a recursive algorithm.
class IMAGEDEV_CPP_API GaussianFilter3d final : public GenericAlgorithm
{
public:
    /// The algorithm implementation used to compute the gaussian filter.
    enum FilterMode
    {
    /// This mode uses a Finite Impulse Response algorithm on the X, Y, and Z directions.
        SEPARABLE = 0,
    /// This mode uses an Infinite Impulse Response algorithm on the X, Y, and Z directions.
        RECURSIVE
    };
    /// The output data type. It can either be the same as the input type, or forced to be float. In the case of floating input images, this parameter has no effect.
    enum OutputType
    {
    /// The output image has same type as the input image. In the case of integer images, this mode can lead to a loss of precision.
        SAME_AS_INPUT = 0,
    /// The output image type is forced to floating point.
        FLOAT_32_BIT
    };

    // Command constructor.
    GaussianFilter3d();


    /// Gets the inputImage parameter.
    /// The input image. The type of image can be integer or float.@BR
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image. The type of image can be integer or float.@BR
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the filterMode parameter.
    /// The algorithm implementation used to compute the gaussian filter.
    GaussianFilter3d::FilterMode filterMode() const;
    /// Sets the filterMode parameter.
    /// The algorithm implementation used to compute the gaussian filter.
    void setFilterMode( const GaussianFilter3d::FilterMode& filterMode );

    /// Gets the standardDeviation parameter.
    /// The standard deviation value for each direction (X, Y, Z) in voxel units. Each value must be greater or equal to 0.0.
    iolink::Vector3d standardDeviation() const;
    /// Sets the standardDeviation parameter.
    /// The standard deviation value for each direction (X, Y, Z) in voxel units. Each value must be greater or equal to 0.0.
    void setStandardDeviation( const iolink::Vector3d& standardDeviation );

    /// Gets the kernelSizeFactor parameter.
    /// This parameter is used to compute the size of the kernel applied in the SEPARABLE mode. The kernel size value is twice the kernelSizeFactor multiplied by the standard deviation associated with the axis. If the resulting kernel size is even, it is incremented by one in order to ensure an odd kernel size. This parameter is ignored in RECURSIVE mode.
    double kernelSizeFactor() const;
    /// Sets the kernelSizeFactor parameter.
    /// This parameter is used to compute the size of the kernel applied in the SEPARABLE mode. The kernel size value is twice the kernelSizeFactor multiplied by the standard deviation associated with the axis. If the resulting kernel size is even, it is incremented by one in order to ensure an odd kernel size. This parameter is ignored in RECURSIVE mode.
    void setKernelSizeFactor( const double& kernelSizeFactor );

    /// Gets the outputType parameter.
    /// The output data type. It can either be the same as the input type, or forced to be float. In the case of floating input images, this parameter has no effect.
    GaussianFilter3d::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The output data type. It can either be the same as the input type, or forced to be float. In the case of floating input images, this parameter has no effect.
    void setOutputType( const GaussianFilter3d::OutputType& outputType );

    /// Gets the lowMemory parameter.
    /// This parameter defines if the SEPARABLE algorithm must limit its memory usage. If equal to false, a temporary 32-bit float image is used to store the result before casting it to the output type. The result is thus less precise and faster to compute when this parameter is set to true. This parameter is ignored in RECURSIVE mode.
    bool lowMemory() const;
    /// Sets the lowMemory parameter.
    /// This parameter defines if the SEPARABLE algorithm must limit its memory usage. If equal to false, a temporary 32-bit float image is used to store the result before casting it to the output type. The result is thus less precise and faster to compute when this parameter is set to true. This parameter is ignored in RECURSIVE mode.
    void setLowMemory( const bool& lowMemory );

    /// Gets the outputImage parameter.
    /// The output image. Dimensions, calibration, and interpretation of the output image are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Dimensions, calibration, and interpretation of the output image are forced to the same values as the input.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Applies a three-dimensional Gaussian filter using either a separable finite kernel, or a recursive algorithm.
/// @param inputImage The input image. The type of image can be integer or float.@BR
/// @param filterMode The algorithm implementation used to compute the gaussian filter.
/// @param standardDeviation The standard deviation value for each direction (X, Y, Z) in voxel units. Each value must be greater or equal to 0.0.
/// @param kernelSizeFactor This parameter is used to compute the size of the kernel applied in the SEPARABLE mode. The kernel size value is twice the kernelSizeFactor multiplied by the standard deviation associated with the axis. If the resulting kernel size is even, it is incremented by one in order to ensure an odd kernel size. This parameter is ignored in RECURSIVE mode.
/// @param outputType The output data type. It can either be the same as the input type, or forced to be float. In the case of floating input images, this parameter has no effect.
/// @param lowMemory This parameter defines if the SEPARABLE algorithm must limit its memory usage. If equal to false, a temporary 32-bit float image is used to store the result before casting it to the output type. The result is thus less precise and faster to compute when this parameter is set to true. This parameter is ignored in RECURSIVE mode.
/// @param outputImage The output image. Dimensions, calibration, and interpretation of the output image are forced to the same values as the input.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
gaussianFilter3d( std::shared_ptr< iolink::ImageView > inputImage,
                  GaussianFilter3d::FilterMode filterMode,
                  const iolink::Vector3d& standardDeviation,
                  double kernelSizeFactor,
                  GaussianFilter3d::OutputType outputType,
                  bool lowMemory,
                  std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
