/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a fast determination of the catchment basins in a grayscale image from a predefined set of markers in a low memory consumption mode.
class IMAGEDEV_CPP_API MarkerBasedWatershedBasinsByBlock final : public GenericAlgorithm
{
public:
    /// The mode for applying the watershed algorithm.
    enum AlgorithmMode
    {
    /// The result is repeatable but slower to compute.
        REPEATABLE = 0,
    /// The result is faster to compute but not repeatable because of asynchronous parallel computation. Since a watershed problem does not generally have a unique solution, two processings of the same image can lead to two different results (both exact).
        FAST
    };

    // Command constructor.
    MarkerBasedWatershedBasinsByBlock();


    /// Gets the inputGrayImage parameter.
    /// The input grayscale image representing the landscape of the watershed.
    std::shared_ptr< iolink::ImageView > inputGrayImage() const;
    /// Sets the inputGrayImage parameter.
    /// The input grayscale image representing the landscape of the watershed.
    void setInputGrayImage( std::shared_ptr< iolink::ImageView > inputGrayImage );

    /// Gets the inputMarkerImage parameter.
    /// The input label image of markers.
    std::shared_ptr< iolink::ImageView > inputMarkerImage() const;
    /// Sets the inputMarkerImage parameter.
    /// The input label image of markers.
    void setInputMarkerImage( std::shared_ptr< iolink::ImageView > inputMarkerImage );

    /// Gets the algorithmMode parameter.
    /// The mode for applying the watershed algorithm.
    MarkerBasedWatershedBasinsByBlock::AlgorithmMode algorithmMode() const;
    /// Sets the algorithmMode parameter.
    /// The mode for applying the watershed algorithm.
    void setAlgorithmMode( const MarkerBasedWatershedBasinsByBlock::AlgorithmMode& algorithmMode );

    /// Gets the blockSize parameter.
    /// The size of blocks used to crop the input images.
    int32_t blockSize() const;
    /// Sets the blockSize parameter.
    /// The size of blocks used to crop the input images.
    void setBlockSize( const int32_t& blockSize );

    /// Gets the outputLabelImage parameter.
    /// The output label image of basins. Its dimensions are forced to the same values as the input image.
    std::shared_ptr< iolink::ImageView > outputLabelImage() const;
    /// Sets the outputLabelImage parameter.
    /// The output label image of basins. Its dimensions are forced to the same values as the input image.
    void setOutputLabelImage( std::shared_ptr< iolink::ImageView > outputLabelImage );

    // Method to launch the command.
    void execute();

};

/// Performs a fast determination of the catchment basins in a grayscale image from a predefined set of markers in a low memory consumption mode.
/// @param inputGrayImage The input grayscale image representing the landscape of the watershed.
/// @param inputMarkerImage The input label image of markers.
/// @param algorithmMode The mode for applying the watershed algorithm.
/// @param blockSize The size of blocks used to crop the input images.
/// @param outputLabelImage The output label image of basins. Its dimensions are forced to the same values as the input image.
/// @return Returns the outputLabelImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
markerBasedWatershedBasinsByBlock( std::shared_ptr< iolink::ImageView > inputGrayImage,
                                   std::shared_ptr< iolink::ImageView > inputMarkerImage,
                                   MarkerBasedWatershedBasinsByBlock::AlgorithmMode algorithmMode,
                                   int32_t blockSize,
                                   std::shared_ptr< iolink::ImageView > outputLabelImage = nullptr );
} // namespace imagedev
