/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Applies a median operator, which is a non-linear smoothing filter, on a two-dimensional image.
class IMAGEDEV_CPP_API MedianFilter2d final : public GenericAlgorithm
{
public:
    /// The structuring element shape and application mode.
    enum KernelMode
    {
    /// This mode considers a square structuring element of side length 2 x kernelRadius + 1 pixels.
        SQUARE = 0,
    /// This mode considers a disk structuring element of diameter 2 x kernelRadius + 1 pixels.
        DISK
    };
    /// Handles all types of method for finding the median.
    enum SearchMode
    {
    /// The algorithm automatically chooses the method to find the median value.
        AUTOMATIC = 0,
    /// The algorithm uses a histogram to find the median value.
        HISTOGRAM,
    /// The algorithm uses a shoting by selection to find the median value.
        SELECTION
    };

    // Command constructor.
    MedianFilter2d();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the kernelRadius parameter.
    /// The half size of the structuring element. A square structuring element always has an odd side length (3x3, 5x5, etc.) which is defined by twice the kernel radius + 1.
    uint32_t kernelRadius() const;
    /// Sets the kernelRadius parameter.
    /// The half size of the structuring element. A square structuring element always has an odd side length (3x3, 5x5, etc.) which is defined by twice the kernel radius + 1.
    void setKernelRadius( const uint32_t& kernelRadius );

    /// Gets the kernelMode parameter.
    /// The structuring element shape and application mode.
    MedianFilter2d::KernelMode kernelMode() const;
    /// Sets the kernelMode parameter.
    /// The structuring element shape and application mode.
    void setKernelMode( const MedianFilter2d::KernelMode& kernelMode );

    /// Gets the searchMode parameter.
    /// Handles all types of method for finding the median.
    MedianFilter2d::SearchMode searchMode() const;
    /// Sets the searchMode parameter.
    /// Handles all types of method for finding the median.
    void setSearchMode( const MedianFilter2d::SearchMode& searchMode );

    /// Gets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Applies a median operator, which is a non-linear smoothing filter, on a two-dimensional image.
/// @param inputImage The input image.
/// @param kernelRadius The half size of the structuring element. A square structuring element always has an odd side length (3x3, 5x5, etc.) which is defined by twice the kernel radius + 1.
/// @param kernelMode The structuring element shape and application mode.
/// @param searchMode Handles all types of method for finding the median.
/// @param outputImage The output image. Its dimensions and type are forced to the same values as the input.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
medianFilter2d( std::shared_ptr< iolink::ImageView > inputImage,
                uint32_t kernelRadius,
                MedianFilter2d::KernelMode kernelMode,
                MedianFilter2d::SearchMode searchMode,
                std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
