/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/Model/OnnxModel.h>

namespace imagedev
{
/// Computes a prediction on a three-dimensional image from an ONNX model and generates an image representing the prediction scores.
class IMAGEDEV_CPP_API OnnxPredictionFiltering3d final : public GenericAlgorithm
{
public:
    /// The tensor layout expected as input by the model. The input image is automatically converted to this layout by the algorithm.
    enum DataFormat
    {
    /// The layout is organized with interlaced channels. For instance, if the input is a color image, each pixel presents its RGB components successively.
        NDHWC = 0,
    /// The layout is organized with separate channels. Each channel is an individual plane.
        NCDHW
    };
    /// The type of normalization to apply before computing the prediction. It is recommended to apply the same pre-processing as during the training.
    enum class InputNormalizationType
    {
    /// No normalization is applied before executing the prediction.
        NONE = 0,
    /// A normalization is applied by subtracting the minimum and dividing by data range.
        MIN_MAX,
    /// A normalization is applied by subtracting the mean and dividing by the standard deviation.
        STANDARDIZATION
    };
    /// The scope for computing normalization (mean, standard deviation, minimum or maximum). This parameter is ignored if the normalization type is set to NONE.
    enum NormalizationScope
    {
    /// The normalization is applied globally on the input batch.
        GLOBAL = 0,
    /// The normalization is applied individually on each image of the input batch.
        PER_VOLUME
    };
    /// The type of normalization to apply after computing the prediction. This parameter is ignored if the input normalization type is set to NONE.
    enum class OutputNormalizationType
    {
    /// No normalization is applied on the output image.
        NONE = 0,
    /// The inverse operation is applied to set the output image in the same range as the input.
        INVERSE
    };
    /// The output data type. It can either be the same as the input type or forced to float. This parameter is ignored if the input normalization type is set to NONE.
    enum OutputType
    {
    /// The output image has the same type as the input image.
        SAME_AS_INPUT = 0,
    /// The output image type is forced to floating point.
        FLOAT_32_BIT
    };

    // Command constructor.
    OnnxPredictionFiltering3d();


    /// Gets the inputImage parameter.
    /// The input image. It can be a grayscale or color image, depending on the selected model.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image. It can be a grayscale or color image, depending on the selected model.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputOnnxModel parameter.
    /// The in memory ONNX model.
    OnnxModel::Ptr inputOnnxModel() const;
    /// Sets the inputOnnxModel parameter.
    /// The in memory ONNX model.
    void setInputOnnxModel( const OnnxModel::Ptr& inputOnnxModel );

    /// Gets the dataFormat parameter.
    /// The tensor layout expected as input by the model. The input image is automatically converted to this layout by the algorithm.
    OnnxPredictionFiltering3d::DataFormat dataFormat() const;
    /// Sets the dataFormat parameter.
    /// The tensor layout expected as input by the model. The input image is automatically converted to this layout by the algorithm.
    void setDataFormat( const OnnxPredictionFiltering3d::DataFormat& dataFormat );

    /// Gets the inputNormalizationType parameter.
    /// The type of normalization to apply before computing the prediction. It is recommended to apply the same pre-processing as during the training.
    OnnxPredictionFiltering3d::InputNormalizationType inputNormalizationType() const;
    /// Sets the inputNormalizationType parameter.
    /// The type of normalization to apply before computing the prediction. It is recommended to apply the same pre-processing as during the training.
    void setInputNormalizationType( const OnnxPredictionFiltering3d::InputNormalizationType& inputNormalizationType );

    /// Gets the normalizationRange parameter.
    /// The data range in which the input image is normalized before computing the prediction. It is recommended to apply the same pre-processing as during the training. This parameter is ignored if the normalization type is set to NONE.
    iolink::Vector2d normalizationRange() const;
    /// Sets the normalizationRange parameter.
    /// The data range in which the input image is normalized before computing the prediction. It is recommended to apply the same pre-processing as during the training. This parameter is ignored if the normalization type is set to NONE.
    void setNormalizationRange( const iolink::Vector2d& normalizationRange );

    /// Gets the normalizationScope parameter.
    /// The scope for computing normalization (mean, standard deviation, minimum or maximum). This parameter is ignored if the normalization type is set to NONE.
    OnnxPredictionFiltering3d::NormalizationScope normalizationScope() const;
    /// Sets the normalizationScope parameter.
    /// The scope for computing normalization (mean, standard deviation, minimum or maximum). This parameter is ignored if the normalization type is set to NONE.
    void setNormalizationScope( const OnnxPredictionFiltering3d::NormalizationScope& normalizationScope );

    /// Gets the tileSize parameter.
    /// The width, height, and depth in pixels of the sliding window. This size includes the user defined tile overlap. It must be a multiple of 2 to the power of the number of downsampling or upsampling layers.
    iolink::Vector3u32 tileSize() const;
    /// Sets the tileSize parameter.
    /// The width, height, and depth in pixels of the sliding window. This size includes the user defined tile overlap. It must be a multiple of 2 to the power of the number of downsampling or upsampling layers.
    void setTileSize( const iolink::Vector3u32& tileSize );

    /// Gets the tileOverlap parameter.
    /// The number of pixels used as overlap between the tiles. An overlap of zero may lead to artifacts in the prediction result. A non-zero overlap reduces such artifacts but increases the computation time.
    uint32_t tileOverlap() const;
    /// Sets the tileOverlap parameter.
    /// The number of pixels used as overlap between the tiles. An overlap of zero may lead to artifacts in the prediction result. A non-zero overlap reduces such artifacts but increases the computation time.
    void setTileOverlap( const uint32_t& tileOverlap );

    /// Gets the outputNormalizationType parameter.
    /// The type of normalization to apply after computing the prediction. This parameter is ignored if the input normalization type is set to NONE.
    OnnxPredictionFiltering3d::OutputNormalizationType outputNormalizationType() const;
    /// Sets the outputNormalizationType parameter.
    /// The type of normalization to apply after computing the prediction. This parameter is ignored if the input normalization type is set to NONE.
    void setOutputNormalizationType( const OnnxPredictionFiltering3d::OutputNormalizationType& outputNormalizationType );

    /// Gets the outputType parameter.
    /// The output data type. It can either be the same as the input type or forced to float. This parameter is ignored if the input normalization type is set to NONE.
    OnnxPredictionFiltering3d::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The output data type. It can either be the same as the input type or forced to float. This parameter is ignored if the input normalization type is set to NONE.
    void setOutputType( const OnnxPredictionFiltering3d::OutputType& outputType );

    /// Gets the outputImage parameter.
    /// The output image. Its spatial dimensions, and calibration are forced to the same values as the input. Its number of channels depends on the selected model. Its type depends on the selected output type.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its spatial dimensions, and calibration are forced to the same values as the input. Its number of channels depends on the selected model. Its type depends on the selected output type.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Computes a prediction on a three-dimensional image from an ONNX model and generates an image representing the prediction scores.
/// @param inputImage The input image. It can be a grayscale or color image, depending on the selected model.
/// @param inputOnnxModel The in memory ONNX model.
/// @param dataFormat The tensor layout expected as input by the model. The input image is automatically converted to this layout by the algorithm.
/// @param inputNormalizationType The type of normalization to apply before computing the prediction. It is recommended to apply the same pre-processing as during the training.
/// @param normalizationRange The data range in which the input image is normalized before computing the prediction. It is recommended to apply the same pre-processing as during the training. This parameter is ignored if the normalization type is set to NONE.
/// @param normalizationScope The scope for computing normalization (mean, standard deviation, minimum or maximum). This parameter is ignored if the normalization type is set to NONE.
/// @param tileSize The width, height, and depth in pixels of the sliding window. This size includes the user defined tile overlap. It must be a multiple of 2 to the power of the number of downsampling or upsampling layers.
/// @param tileOverlap The number of pixels used as overlap between the tiles. An overlap of zero may lead to artifacts in the prediction result. A non-zero overlap reduces such artifacts but increases the computation time.
/// @param outputNormalizationType The type of normalization to apply after computing the prediction. This parameter is ignored if the input normalization type is set to NONE.
/// @param outputType The output data type. It can either be the same as the input type or forced to float. This parameter is ignored if the input normalization type is set to NONE.
/// @param outputImage The output image. Its spatial dimensions, and calibration are forced to the same values as the input. Its number of channels depends on the selected model. Its type depends on the selected output type.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
onnxPredictionFiltering3d( std::shared_ptr< iolink::ImageView > inputImage,
                           OnnxModel::Ptr inputOnnxModel,
                           OnnxPredictionFiltering3d::DataFormat dataFormat,
                           OnnxPredictionFiltering3d::InputNormalizationType inputNormalizationType,
                           const iolink::Vector2d& normalizationRange,
                           OnnxPredictionFiltering3d::NormalizationScope normalizationScope,
                           const iolink::Vector3u32& tileSize,
                           uint32_t tileOverlap,
                           OnnxPredictionFiltering3d::OutputNormalizationType outputNormalizationType,
                           OnnxPredictionFiltering3d::OutputType outputType,
                           std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
