/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/GlobalMeasurements/LocalOrientation3dMsr.h>

namespace imagedev
{
// Output structure of the orientationMapFourier3d function.
struct OrientationMapFourier3dOutput
{
    /// The grayscale output image representing the oriented sticks computed in each block. Intensities correspond to the confidence of the estimated orientation multplied by 100. Image dimensions are forced to the same values as the input. Its type is unsigned 8-bit integer.
    std::shared_ptr< iolink::ImageView > outputImage;
    /// The output measurement result for each block.
    LocalOrientation3dMsr::Ptr outputMeasurement;
};

/// Determines a block-wise orientation of structures contained in a three-dimensional grayscale image.
class IMAGEDEV_CPP_API OrientationMapFourier3d final : public GenericAlgorithm
{
public:
    /// The policy for splitting the image into blocks.
    enum BlockOverlap
    {
    /// The algorithm splits the image into contiguous blocks. The offset between two consecutive block origins is equal to the block size.
        NO = 0,
    /// The algorithm splits the image into overlapping blocks. The offset between two consecutive block origins is equal to the half of the block size.
        YES
    };

    // Command constructor.
    OrientationMapFourier3d();


    /// Gets the inputImage parameter.
    /// The grayscale input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The grayscale input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the blockSize parameter.
    /// The side size in pixels of the cubic blocks.
    int32_t blockSize() const;
    /// Sets the blockSize parameter.
    /// The side size in pixels of the cubic blocks.
    void setBlockSize( const int32_t& blockSize );

    /// Gets the blockOverlap parameter.
    /// The policy for splitting the image into blocks.
    OrientationMapFourier3d::BlockOverlap blockOverlap() const;
    /// Sets the blockOverlap parameter.
    /// The policy for splitting the image into blocks.
    void setBlockOverlap( const OrientationMapFourier3d::BlockOverlap& blockOverlap );

    /// Gets the minThreshold parameter.
    /// The minimum threshold value applied to the input grayscale image. Voxels having an intensity lower than this threshold are not considered by the algorithm.
    int32_t minThreshold() const;
    /// Sets the minThreshold parameter.
    /// The minimum threshold value applied to the input grayscale image. Voxels having an intensity lower than this threshold are not considered by the algorithm.
    void setMinThreshold( const int32_t& minThreshold );

    /// Gets the maxThreshold parameter.
    /// The maximum threshold value applied to the input grayscale image. Voxels having an intensity greater than this threshold are not considered by the algorithm.
    int32_t maxThreshold() const;
    /// Sets the maxThreshold parameter.
    /// The maximum threshold value applied to the input grayscale image. Voxels having an intensity greater than this threshold are not considered by the algorithm.
    void setMaxThreshold( const int32_t& maxThreshold );

    /// Gets the outputImage parameter.
    /// The grayscale output image representing the oriented sticks computed in each block. Intensities correspond to the confidence of the estimated orientation multplied by 100. Image dimensions are forced to the same values as the input. Its type is unsigned 8-bit integer.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The grayscale output image representing the oriented sticks computed in each block. Intensities correspond to the confidence of the estimated orientation multplied by 100. Image dimensions are forced to the same values as the input. Its type is unsigned 8-bit integer.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    /// Gets the outputMeasurement parameter.
    /// The output measurement result for each block.
    LocalOrientation3dMsr::Ptr outputMeasurement() const;

    // Method to launch the command.
    void execute();

};

/// Determines a block-wise orientation of structures contained in a three-dimensional grayscale image.
/// @param inputImage The grayscale input image.
/// @param blockSize The side size in pixels of the cubic blocks.
/// @param blockOverlap The policy for splitting the image into blocks.
/// @param minThreshold The minimum threshold value applied to the input grayscale image. Voxels having an intensity lower than this threshold are not considered by the algorithm.
/// @param maxThreshold The maximum threshold value applied to the input grayscale image. Voxels having an intensity greater than this threshold are not considered by the algorithm.
/// @param outputImage The grayscale output image representing the oriented sticks computed in each block. Intensities correspond to the confidence of the estimated orientation multplied by 100. Image dimensions are forced to the same values as the input. Its type is unsigned 8-bit integer.
/// @param outputMeasurement The output measurement result for each block.
/// @return Returns a OrientationMapFourier3dOutput structure containing the outputImage and outputMeasurement output parameters.
IMAGEDEV_CPP_API 
OrientationMapFourier3dOutput
orientationMapFourier3d( std::shared_ptr< iolink::ImageView > inputImage,
                         int32_t blockSize,
                         OrientationMapFourier3d::BlockOverlap blockOverlap,
                         int32_t minThreshold,
                         int32_t maxThreshold,
                         std::shared_ptr< iolink::ImageView > outputImage = nullptr,
                         LocalOrientation3dMsr::Ptr outputMeasurement = nullptr );
} // namespace imagedev
