/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Computes a point spread function (PSF) for deconvolution of widefield and confocal microscopic image data.
class IMAGEDEV_CPP_API PointSpreadFunctionImage3d final : public GenericAlgorithm
{
public:
    /// Specifies whether the PSF of a widefield microscope or of a confocal microscope should be computed.
    enum MicroscopicMode
    {
    /// Compute the PSF of a widefield microscope.
        WIDEFIELD = 0,
    /// Compute the PSF of a confocal microscope.
        CONFOCAL
    };

    // Command constructor.
    PointSpreadFunctionImage3d();


    /// Gets the numericalAperture parameter.
    /// The numerical aperture of the microscope.
    double numericalAperture() const;
    /// Sets the numericalAperture parameter.
    /// The numerical aperture of the microscope.
    void setNumericalAperture( const double& numericalAperture );

    /// Gets the wavelength parameter.
    /// The wavelength (as measured in a vacuum) of the emitted light in nanometers.
    double wavelength() const;
    /// Sets the wavelength parameter.
    /// The wavelength (as measured in a vacuum) of the emitted light in nanometers.
    void setWavelength( const double& wavelength );

    /// Gets the index parameter.
    /// The refractive index of the specimen.
    double index() const;
    /// Sets the index parameter.
    /// The refractive index of the specimen.
    void setIndex( const double& index );

    /// Gets the microscopicMode parameter.
    /// Specifies whether the PSF of a widefield microscope or of a confocal microscope should be computed.
    PointSpreadFunctionImage3d::MicroscopicMode microscopicMode() const;
    /// Sets the microscopicMode parameter.
    /// Specifies whether the PSF of a widefield microscope or of a confocal microscope should be computed.
    void setMicroscopicMode( const PointSpreadFunctionImage3d::MicroscopicMode& microscopicMode );

    /// Gets the shape parameter.
    /// The X, Y, and Z dimensions, in pixels, of the output image.
    iolink::Vector3u32 shape() const;
    /// Sets the shape parameter.
    /// The X, Y, and Z dimensions, in pixels, of the output image.
    void setShape( const iolink::Vector3u32& shape );

    /// Gets the spacing parameter.
    /// The X, Y, and Z voxel size in nanometers of the PSF image to be generated.
    iolink::Vector3d spacing() const;
    /// Sets the spacing parameter.
    /// The X, Y, and Z voxel size in nanometers of the PSF image to be generated.
    void setSpacing( const iolink::Vector3d& spacing );

    /// Gets the outputImage parameter.
    /// The output image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Computes a point spread function (PSF) for deconvolution of widefield and confocal microscopic image data.
/// @param numericalAperture The numerical aperture of the microscope.
/// @param wavelength The wavelength (as measured in a vacuum) of the emitted light in nanometers.
/// @param index The refractive index of the specimen.
/// @param microscopicMode Specifies whether the PSF of a widefield microscope or of a confocal microscope should be computed.
/// @param shape The X, Y, and Z dimensions, in pixels, of the output image.
/// @param spacing The X, Y, and Z voxel size in nanometers of the PSF image to be generated.
/// @param outputImage The output image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
pointSpreadFunctionImage3d( double numericalAperture,
                            double wavelength,
                            double index,
                            PointSpreadFunctionImage3d::MicroscopicMode microscopicMode,
                            const iolink::Vector3u32& shape,
                            const iolink::Vector3d& spacing,
                            std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
