/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Computes a radial background image from a grayscale image.
class IMAGEDEV_CPP_API RadialBackgroundImage2d final : public GenericAlgorithm
{
public:
    /// The way to estimate the background model of the Fourier transform module.
    enum BackgroundModel
    {
    /// The background is estimated as a radial average of the Fourier transform of the whole image.
        AVERAGE = 0,
    /// The background is estimated by fitting a polynomial of degree one on the radial average of the masked image.
        POLYNOMIAL_ORDER1,
    /// The background is estimated by fitting a polynomial of degree two on the radial average of the masked image.
        POLYNOMIAL_ORDER2,
    /// The background is estimated by fitting a polynomial of degree three on the radial average of the masked image.
        POLYNOMIAL_ORDER3,
    /// The background is estimated by fitting a polynomial of degree four on the radial average of the masked image.
        POLYNOMIAL_ORDER4,
    /// The background is estimated by fitting a polynomial of degree five on the radial average of the masked image.
        POLYNOMIAL_ORDER5,
    /// The background is estimated by fitting a polynomial of degree six on the radial average of the masked image.
        POLYNOMIAL_ORDER6
    };
    /// The way to define the center of the radial profiles.
    enum CenterMode
    {
    /// The center of the radial profiles is the center of the input image.
        IMAGE_CENTER = 0,
    /// The center of the radial profiles is user-defined.
        OTHER
    };

    // Command constructor.
    RadialBackgroundImage2d();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputMaskImage parameter.
    /// The binary mask image for the polynomial interpolation. If it equals null, the background is computed from all pixels of the input image. This image must have same dimensions as the input image.
    std::shared_ptr< iolink::ImageView > inputMaskImage() const;
    /// Sets the inputMaskImage parameter.
    /// The binary mask image for the polynomial interpolation. If it equals null, the background is computed from all pixels of the input image. This image must have same dimensions as the input image.
    void setInputMaskImage( std::shared_ptr< iolink::ImageView > inputMaskImage );

    /// Gets the backgroundModel parameter.
    /// The way to estimate the background model of the Fourier transform module.
    RadialBackgroundImage2d::BackgroundModel backgroundModel() const;
    /// Sets the backgroundModel parameter.
    /// The way to estimate the background model of the Fourier transform module.
    void setBackgroundModel( const RadialBackgroundImage2d::BackgroundModel& backgroundModel );

    /// Gets the centerMode parameter.
    /// The way to define the center of the radial profiles.
    RadialBackgroundImage2d::CenterMode centerMode() const;
    /// Sets the centerMode parameter.
    /// The way to define the center of the radial profiles.
    void setCenterMode( const RadialBackgroundImage2d::CenterMode& centerMode );

    /// Gets the centerPoint parameter.
    /// The X and Y coordinates used as center if the center mode parameter is set to OTHER. This parameter is ignored in IMAGE_CENTER mode.
    iolink::Vector2i32 centerPoint() const;
    /// Sets the centerPoint parameter.
    /// The X and Y coordinates used as center if the center mode parameter is set to OTHER. This parameter is ignored in IMAGE_CENTER mode.
    void setCenterPoint( const iolink::Vector2i32& centerPoint );

    /// Gets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its dimensions and type are forced to the same values as the input.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Computes a radial background image from a grayscale image.
/// @param inputImage The input image.
/// @param inputMaskImage The binary mask image for the polynomial interpolation. If it equals null, the background is computed from all pixels of the input image. This image must have same dimensions as the input image.
/// @param backgroundModel The way to estimate the background model of the Fourier transform module.
/// @param centerMode The way to define the center of the radial profiles.
/// @param centerPoint The X and Y coordinates used as center if the center mode parameter is set to OTHER. This parameter is ignored in IMAGE_CENTER mode.
/// @param outputImage The output image. Its dimensions and type are forced to the same values as the input.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
radialBackgroundImage2d( std::shared_ptr< iolink::ImageView > inputImage,
                         std::shared_ptr< iolink::ImageView > inputMaskImage,
                         RadialBackgroundImage2d::BackgroundModel backgroundModel,
                         RadialBackgroundImage2d::CenterMode centerMode,
                         const iolink::Vector2i32& centerPoint,
                         std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
