/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the radialGradientCentered2d function.
struct RadialGradientCentered2dOutput
{
    /// The normal gradient output image.
    std::shared_ptr< iolink::ImageView > outputNormalImage;
    /// The tangent gradient output image.
    std::shared_ptr< iolink::ImageView > outputTangentImage;
};

/// Performs a two-dimensional gradient projection with a directional vector defined by a center point.
class IMAGEDEV_CPP_API RadialGradientCentered2d final : public GenericAlgorithm
{
public:
    /// The output images to be computed.
    enum ProjectionMode
    {
    /// The engine computes the normal gradient. Only the outputNormalImage output is set using this mode.
        GRADIENT_NORMAL = 0,
    /// The engine computes the normal and tangential gradients. Both outputNormalImage and outputTangentImage outputs are set using this mode.
        GRADIENT_NORMAL_TANGENTIAL
    };

    // Command constructor.
    RadialGradientCentered2d();


    /// Gets the inputImageX parameter.
    /// The X-gradient input image.
    std::shared_ptr< iolink::ImageView > inputImageX() const;
    /// Sets the inputImageX parameter.
    /// The X-gradient input image.
    void setInputImageX( std::shared_ptr< iolink::ImageView > inputImageX );

    /// Gets the inputImageY parameter.
    /// The Y-gradient input image.
    std::shared_ptr< iolink::ImageView > inputImageY() const;
    /// Sets the inputImageY parameter.
    /// The Y-gradient input image.
    void setInputImageY( std::shared_ptr< iolink::ImageView > inputImageY );

    /// Gets the projectionCenter parameter.
    /// The center of projection coordinates.
    iolink::Vector2i32 projectionCenter() const;
    /// Sets the projectionCenter parameter.
    /// The center of projection coordinates.
    void setProjectionCenter( const iolink::Vector2i32& projectionCenter );

    /// Gets the projectionMode parameter.
    /// The output images to be computed.
    RadialGradientCentered2d::ProjectionMode projectionMode() const;
    /// Sets the projectionMode parameter.
    /// The output images to be computed.
    void setProjectionMode( const RadialGradientCentered2d::ProjectionMode& projectionMode );

    /// Gets the outputNormalImage parameter.
    /// The normal gradient output image.
    std::shared_ptr< iolink::ImageView > outputNormalImage() const;
    /// Sets the outputNormalImage parameter.
    /// The normal gradient output image.
    void setOutputNormalImage( std::shared_ptr< iolink::ImageView > outputNormalImage );

    /// Gets the outputTangentImage parameter.
    /// The tangent gradient output image.
    std::shared_ptr< iolink::ImageView > outputTangentImage() const;
    /// Sets the outputTangentImage parameter.
    /// The tangent gradient output image.
    void setOutputTangentImage( std::shared_ptr< iolink::ImageView > outputTangentImage );

    // Method to launch the command.
    void execute();

};

/// Performs a two-dimensional gradient projection with a directional vector defined by a center point.
/// @param inputImageX The X-gradient input image.
/// @param inputImageY The Y-gradient input image.
/// @param projectionCenter The center of projection coordinates.
/// @param projectionMode The output images to be computed.
/// @param outputNormalImage The normal gradient output image.
/// @param outputTangentImage The tangent gradient output image.
/// @return Returns a RadialGradientCentered2dOutput structure containing the outputNormalImage and outputTangentImage output parameters.
IMAGEDEV_CPP_API 
RadialGradientCentered2dOutput
radialGradientCentered2d( std::shared_ptr< iolink::ImageView > inputImageX,
                          std::shared_ptr< iolink::ImageView > inputImageY,
                          const iolink::Vector2i32& projectionCenter,
                          RadialGradientCentered2d::ProjectionMode projectionMode,
                          std::shared_ptr< iolink::ImageView > outputNormalImage = nullptr,
                          std::shared_ptr< iolink::ImageView > outputTangentImage = nullptr );
} // namespace imagedev
