/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Removes ring artifacts from a Computed Tomography volume.
class IMAGEDEV_CPP_API RingArtifactRemoval3d final : public GenericAlgorithm
{
public:
    /// The way to define the center of the cylindrical sample.
    enum CenterMode
    {
    /// The center of the rings is the center of the input image.
        IMAGE_CENTER = 0,
    /// The center of the rings is user-defined.
        OTHER
    };

    // Command constructor.
    RingArtifactRemoval3d();


    /// Gets the inputImage parameter.
    /// The 3D input image to correct.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The 3D input image to correct.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the centerMode parameter.
    /// The way to define the center of the cylindrical sample.
    RingArtifactRemoval3d::CenterMode centerMode() const;
    /// Sets the centerMode parameter.
    /// The way to define the center of the cylindrical sample.
    void setCenterMode( const RingArtifactRemoval3d::CenterMode& centerMode );

    /// Gets the center parameter.
    /// The X and Y coordinates center, defined in voxel units, for each volume slice of the concentric rings to remove. This parameter is ignored if the center mode is set to IMAGE_CENTER.
    iolink::Vector2u32 center() const;
    /// Sets the center parameter.
    /// The X and Y coordinates center, defined in voxel units, for each volume slice of the concentric rings to remove. This parameter is ignored if the center mode is set to IMAGE_CENTER.
    void setCenter( const iolink::Vector2u32& center );

    /// Gets the thresholdRange parameter.
    /// Only voxel intensities inside this range are considered to compute the mean values used by the correction. Other values are processed anyway by the algorithm.
    iolink::Vector2d thresholdRange() const;
    /// Sets the thresholdRange parameter.
    /// Only voxel intensities inside this range are considered to compute the mean values used by the correction. Other values are processed anyway by the algorithm.
    void setThresholdRange( const iolink::Vector2d& thresholdRange );

    /// Gets the radiusRange parameter.
    /// The minimum and maximum radius, in voxel units, defining a hollow cylinder along the Z axis, which is used to compute the whole sample mean value.
    iolink::Vector2u32 radiusRange() const;
    /// Sets the radiusRange parameter.
    /// The minimum and maximum radius, in voxel units, defining a hollow cylinder along the Z axis, which is used to compute the whole sample mean value.
    void setRadiusRange( const iolink::Vector2u32& radiusRange );

    /// Gets the ignoreThresholdsForRings parameter.
    /// Enable or disable the coverPercentage parameter.
    bool ignoreThresholdsForRings() const;
    /// Sets the ignoreThresholdsForRings parameter.
    /// Enable or disable the coverPercentage parameter.
    void setIgnoreThresholdsForRings( const bool& ignoreThresholdsForRings );

    /// Gets the coverPercentage parameter.
    /// If the percentage of voxels between the thresholds in any one ring is less than this value, then all the voxels in the ring are used in the calculation. This parameter is ignored if ignoreThresholdsForRings is disabled.
    double coverPercentage() const;
    /// Sets the coverPercentage parameter.
    /// If the percentage of voxels between the thresholds in any one ring is less than this value, then all the voxels in the ring are used in the calculation. This parameter is ignored if ignoreThresholdsForRings is disabled.
    void setCoverPercentage( const double& coverPercentage );

    /// Gets the singleRingMode parameter.
    /// Option to remove only one user-defined ring.
    bool singleRingMode() const;
    /// Sets the singleRingMode parameter.
    /// Option to remove only one user-defined ring.
    void setSingleRingMode( const bool& singleRingMode );

    /// Gets the singleRingRadiusRange parameter.
    /// The minimum and maximum radius, in voxel units, of the ring to remove. This parameter is ignored if the single ring mode is disabled.
    iolink::Vector2u32 singleRingRadiusRange() const;
    /// Sets the singleRingRadiusRange parameter.
    /// The minimum and maximum radius, in voxel units, of the ring to remove. This parameter is ignored if the single ring mode is disabled.
    void setSingleRingRadiusRange( const iolink::Vector2u32& singleRingRadiusRange );

    /// Gets the singleRingSliceRange parameter.
    /// The first and last slice index where the ring to remove is visible. This parameter is ignored if the single ring mode is disabled.
    iolink::Vector2u32 singleRingSliceRange() const;
    /// Sets the singleRingSliceRange parameter.
    /// The first and last slice index where the ring to remove is visible. This parameter is ignored if the single ring mode is disabled.
    void setSingleRingSliceRange( const iolink::Vector2u32& singleRingSliceRange );

    /// Gets the outputImage parameter.
    /// The corrected image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The corrected image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Removes ring artifacts from a Computed Tomography volume.
/// @param inputImage The 3D input image to correct.
/// @param centerMode The way to define the center of the cylindrical sample.
/// @param center The X and Y coordinates center, defined in voxel units, for each volume slice of the concentric rings to remove. This parameter is ignored if the center mode is set to IMAGE_CENTER.
/// @param thresholdRange Only voxel intensities inside this range are considered to compute the mean values used by the correction. Other values are processed anyway by the algorithm.
/// @param radiusRange The minimum and maximum radius, in voxel units, defining a hollow cylinder along the Z axis, which is used to compute the whole sample mean value.
/// @param ignoreThresholdsForRings Enable or disable the coverPercentage parameter.
/// @param coverPercentage If the percentage of voxels between the thresholds in any one ring is less than this value, then all the voxels in the ring are used in the calculation. This parameter is ignored if ignoreThresholdsForRings is disabled.
/// @param singleRingMode Option to remove only one user-defined ring.
/// @param singleRingRadiusRange The minimum and maximum radius, in voxel units, of the ring to remove. This parameter is ignored if the single ring mode is disabled.
/// @param singleRingSliceRange The first and last slice index where the ring to remove is visible. This parameter is ignored if the single ring mode is disabled.
/// @param outputImage The corrected image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
ringArtifactRemoval3d( std::shared_ptr< iolink::ImageView > inputImage,
                       RingArtifactRemoval3d::CenterMode centerMode,
                       const iolink::Vector2u32& center,
                       const iolink::Vector2d& thresholdRange,
                       const iolink::Vector2u32& radiusRange,
                       bool ignoreThresholdsForRings,
                       double coverPercentage,
                       bool singleRingMode,
                       const iolink::Vector2u32& singleRingRadiusRange,
                       const iolink::Vector2u32& singleRingSliceRange,
                       std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
