/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a linear scaling of the gray level values of each slice of an image stack.
class IMAGEDEV_CPP_API StackRescaleIntensity final : public GenericAlgorithm
{
public:
    /// The type of rescaling to apply on each slice.
    enum ScalingMode
    {
    /// The scaling is performed by subtracting the mean and dividing by the standard deviation.
        STANDARDIZATION = 0,
    /// The scaling is performed by subtracting the minimum and dividing by data range.
        NORMALIZATION
    };
    /// The way to define the input intensity range, applied on each slice of the stack.
    enum RangeMode
    {
    /// The input range is the minimum and maximum of each slice.
        MIN_MAX = 0,
    /// The input range is defined by two user-defined percentiles, computed on each slice.
        PERCENTILE,
    /// The input range is between user-defined bounds [a,b].
        OTHER
    };
    /// The scope for computing the output parameters (mean, standard deviation, percentiles, minimum or maximum).
    enum ScalingScope
    {
    /// The output parameters are computed globally on the input stack.
        GLOBAL = 0,
    /// The output parameters are computed independently on each slice of the stack, and then averaged.
        SLICE_AVERAGING
    };
    /// The output image data type.
    enum OutputType
    {
    /// The output image data type is 1 byte depth. Its possible intensities are unsigned integer from 0 to 255.
        UNSIGNED_INTEGER_8_BIT = 0,
    /// The output image data type is 2 bytes depth. Its possible intensities are unsigned integer from 0 to 65,535.
        UNSIGNED_INTEGER_16_BIT,
    /// The output image data type is 4 bytes depth. Its possible intensities are signed integer from 0 to 4,294,967,295.
        UNSIGNED_INTEGER_32_BIT,
    /// The output image data type is 1 byte depth. Its possible intensities are signed integer from -128 to 127.
        SIGNED_INTEGER_8_BIT,
    /// The output image data type is 2 bytes depth. Its possible intensities are signed integer from -32,768 to 32,767.
        SIGNED_INTEGER_16_BIT,
    /// The output image data type is 4 bytes depth. Its possible intensities are signed integer from -2,147,483,648 to 2,147,483,647.
        SIGNED_INTEGER_32_BIT,
    /// The output image data type is 4 bytes depth. Its possible intensities are unsigned floating-point from -3.4E38 to 3.4E38.
        FLOAT_32_BIT,
    /// The output image data type is 8 bytes depth. Its possible intensities are unsigned floating-point from -1.7E308 to 1.7E308.
        FLOAT_64_BIT,
    /// The output image has the same type as the input image.
        SAME_AS_INPUT
    };

    // Command constructor.
    StackRescaleIntensity();


    /// Gets the inputImage parameter.
    /// The input image stack to normalize. If the input is a 3D volume, it will be considered as a stack of XY images.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image stack to normalize. If the input is a 3D volume, it will be considered as a stack of XY images.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the scalingMode parameter.
    /// The type of rescaling to apply on each slice.
    StackRescaleIntensity::ScalingMode scalingMode() const;
    /// Sets the scalingMode parameter.
    /// The type of rescaling to apply on each slice.
    void setScalingMode( const StackRescaleIntensity::ScalingMode& scalingMode );

    /// Gets the inputRangeMode parameter.
    /// The way to define the input intensity range, applied on each slice of the stack.
    StackRescaleIntensity::RangeMode inputRangeMode() const;
    /// Sets the inputRangeMode parameter.
    /// The way to define the input intensity range, applied on each slice of the stack.
    void setInputRangeMode( const StackRescaleIntensity::RangeMode& inputRangeMode );

    /// Gets the percentageInputRange parameter.
    /// The low and high histogram percentile to use when the input range mode is set to PERCENTILE (in percent). This parameter is ignored with other input range modes.
    iolink::Vector2d percentageInputRange() const;
    /// Sets the percentageInputRange parameter.
    /// The low and high histogram percentile to use when the input range mode is set to PERCENTILE (in percent). This parameter is ignored with other input range modes.
    void setPercentageInputRange( const iolink::Vector2d& percentageInputRange );

    /// Gets the intensityInputRange parameter.
    /// The input intensity range [a, b] to use when the input range mode is set to OTHER. This parameter is ignored with other input range modes.
    iolink::Vector2d intensityInputRange() const;
    /// Sets the intensityInputRange parameter.
    /// The input intensity range [a, b] to use when the input range mode is set to OTHER. This parameter is ignored with other input range modes.
    void setIntensityInputRange( const iolink::Vector2d& intensityInputRange );

    /// Gets the scalingScope parameter.
    /// The scope for computing the output parameters (mean, standard deviation, percentiles, minimum or maximum).
    StackRescaleIntensity::ScalingScope scalingScope() const;
    /// Sets the scalingScope parameter.
    /// The scope for computing the output parameters (mean, standard deviation, percentiles, minimum or maximum).
    void setScalingScope( const StackRescaleIntensity::ScalingScope& scalingScope );

    /// Gets the outputRangeMode parameter.
    /// The way to define the destination intensity range.
    StackRescaleIntensity::RangeMode outputRangeMode() const;
    /// Sets the outputRangeMode parameter.
    /// The way to define the destination intensity range.
    void setOutputRangeMode( const StackRescaleIntensity::RangeMode& outputRangeMode );

    /// Gets the percentageOutputRange parameter.
    /// The low and high histogram percentile to use when the output range mode is set to PERCENTILE (in percent). This parameter is ignored with other output range modes.
    iolink::Vector2d percentageOutputRange() const;
    /// Sets the percentageOutputRange parameter.
    /// The low and high histogram percentile to use when the output range mode is set to PERCENTILE (in percent). This parameter is ignored with other output range modes.
    void setPercentageOutputRange( const iolink::Vector2d& percentageOutputRange );

    /// Gets the intensityOutputRange parameter.
    /// The output minimum and maximum values used in normalization mode. This parameter is disabled in standardization mode and is only enabled if outputRangeMode is set to OTHER.
    iolink::Vector2d intensityOutputRange() const;
    /// Sets the intensityOutputRange parameter.
    /// The output minimum and maximum values used in normalization mode. This parameter is disabled in standardization mode and is only enabled if outputRangeMode is set to OTHER.
    void setIntensityOutputRange( const iolink::Vector2d& intensityOutputRange );

    /// Gets the intensityOutputStatistics parameter.
    /// The mean and standard deviation values used in standardization mode. This parameter is disabled in normalization mode and is only enabled if outputRangeMode is set to OTHER.
    iolink::Vector2d intensityOutputStatistics() const;
    /// Sets the intensityOutputStatistics parameter.
    /// The mean and standard deviation values used in standardization mode. This parameter is disabled in normalization mode and is only enabled if outputRangeMode is set to OTHER.
    void setIntensityOutputStatistics( const iolink::Vector2d& intensityOutputStatistics );

    /// Gets the outputType parameter.
    /// The output image data type.
    StackRescaleIntensity::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The output image data type.
    void setOutputType( const StackRescaleIntensity::OutputType& outputType );

    /// Gets the outputImage parameter.
    /// The output image. Its dimensions are forced to the same values as the input. Its data type is user-defined.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its dimensions are forced to the same values as the input. Its data type is user-defined.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Performs a linear scaling of the gray level values of each slice of an image stack.
/// @param inputImage The input image stack to normalize. If the input is a 3D volume, it will be considered as a stack of XY images.
/// @param scalingMode The type of rescaling to apply on each slice.
/// @param inputRangeMode The way to define the input intensity range, applied on each slice of the stack.
/// @param percentageInputRange The low and high histogram percentile to use when the input range mode is set to PERCENTILE (in percent). This parameter is ignored with other input range modes.
/// @param intensityInputRange The input intensity range [a, b] to use when the input range mode is set to OTHER. This parameter is ignored with other input range modes.
/// @param scalingScope The scope for computing the output parameters (mean, standard deviation, percentiles, minimum or maximum).
/// @param outputRangeMode The way to define the destination intensity range.
/// @param percentageOutputRange The low and high histogram percentile to use when the output range mode is set to PERCENTILE (in percent). This parameter is ignored with other output range modes.
/// @param intensityOutputRange The output minimum and maximum values used in normalization mode. This parameter is disabled in standardization mode and is only enabled if outputRangeMode is set to OTHER.
/// @param intensityOutputStatistics The mean and standard deviation values used in standardization mode. This parameter is disabled in normalization mode and is only enabled if outputRangeMode is set to OTHER.
/// @param outputType The output image data type.
/// @param outputImage The output image. Its dimensions are forced to the same values as the input. Its data type is user-defined.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
stackRescaleIntensity( std::shared_ptr< iolink::ImageView > inputImage,
                       StackRescaleIntensity::ScalingMode scalingMode,
                       StackRescaleIntensity::RangeMode inputRangeMode,
                       const iolink::Vector2d& percentageInputRange,
                       const iolink::Vector2d& intensityInputRange,
                       StackRescaleIntensity::ScalingScope scalingScope,
                       StackRescaleIntensity::RangeMode outputRangeMode,
                       const iolink::Vector2d& percentageOutputRange,
                       const iolink::Vector2d& intensityOutputRange,
                       const iolink::Vector2d& intensityOutputStatistics,
                       StackRescaleIntensity::OutputType outputType,
                       std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
