/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/ArrayX.h>
#include <memory>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Adjusts the positions of the vertices of a structured quadrilateral mesh according to weights provided by a volume.
class IMAGEDEV_CPP_API SurfaceFitting3d final : public GenericAlgorithm
{
public:
    /// The criterion according to which a vertex can be kept as the best fit.
    enum FittingCriterion
    {
    /// The surface is fitted on the intensity image maxima.
        MAXIMUM = 0,
    /// The surface is fitted on the intensity image minima.
        MINIMUM
    };
    /// The type of boundary conditions to apply on the top of the mesh.
    enum BoundaryCondition
    {
    /// The Dirichlet boundary conditions are enabled for the top border. Displacements of vertices are clamped to the value defined by the parameter dirichletValues.
        DIRICHLET = 0,
    /// The Neumann boundary conditions are enabled for the top border. The variation of displacements in the direction perpendicular to the border is null. It means that the displacement of a vertex at the border is equal to the displacement of its direct bottom neighbor.
        NEUMANN
    };
    /// The type of regularization used to smooth the mesh.
    enum RegularizationType
    {
    /// The mesh is smoothed by a classical Laplacian operator. This type of regularization is subject to a "shrinking effect": the surface can lie slightly below the best score points (detected along the normals).
        LAPLACIAN = 0,
    /// The mesh is smoothed by a second order Laplacian operator. This type of regularization does not suffer from the shrinking effect but can be less stable.
        LAPLACIAN_2
    };

    // Command constructor.
    SurfaceFitting3d();


    /// Gets the inputIntensityImage parameter.
    /// The 3D input image containing the weight values to consider for the fitting.
    std::shared_ptr< iolink::ImageView > inputIntensityImage() const;
    /// Sets the inputIntensityImage parameter.
    /// The 3D input image containing the weight values to consider for the fitting.
    void setInputIntensityImage( std::shared_ptr< iolink::ImageView > inputIntensityImage );

    /// Gets the inputSurface parameter.
    /// The input mesh that represents the surface to fit. This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
    std::shared_ptr<iolink::ArrayXd> inputSurface() const;
    /// Sets the inputSurface parameter.
    /// The input mesh that represents the surface to fit. This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
    void setInputSurface( std::shared_ptr<iolink::ArrayXd> inputSurface );

    /// Gets the directions parameter.
    /// The search direction for each vertex.
    std::shared_ptr<iolink::ArrayXd> directions() const;
    /// Sets the directions parameter.
    /// The search direction for each vertex.
    void setDirections( std::shared_ptr<iolink::ArrayXd> directions );

    /// Gets the distance parameter.
    /// The maximum distance to which the vertices can be moved.
    double distance() const;
    /// Sets the distance parameter.
    /// The maximum distance to which the vertices can be moved.
    void setDistance( const double& distance );

    /// Gets the thresholdRange parameter.
    /// The range of values that can be considered for the best fit.
    iolink::Vector2d thresholdRange() const;
    /// Sets the thresholdRange parameter.
    /// The range of values that can be considered for the best fit.
    void setThresholdRange( const iolink::Vector2d& thresholdRange );

    /// Gets the smoothing parameter.
    /// The strength of the regularization applied on the fitted mesh.
    double smoothing() const;
    /// Sets the smoothing parameter.
    /// The strength of the regularization applied on the fitted mesh.
    void setSmoothing( const double& smoothing );

    /// Gets the fittingCriterion parameter.
    /// The criterion according to which a vertex can be kept as the best fit.
    SurfaceFitting3d::FittingCriterion fittingCriterion() const;
    /// Sets the fittingCriterion parameter.
    /// The criterion according to which a vertex can be kept as the best fit.
    void setFittingCriterion( const SurfaceFitting3d::FittingCriterion& fittingCriterion );

    /// Gets the topBoundaryCondition parameter.
    /// The type of boundary conditions to apply on the top of the mesh.
    SurfaceFitting3d::BoundaryCondition topBoundaryCondition() const;
    /// Sets the topBoundaryCondition parameter.
    /// The type of boundary conditions to apply on the top of the mesh.
    void setTopBoundaryCondition( const SurfaceFitting3d::BoundaryCondition& topBoundaryCondition );

    /// Gets the bottomBoundaryCondition parameter.
    /// The type of boundary conditions to apply on the bottom of the mesh.
    SurfaceFitting3d::BoundaryCondition bottomBoundaryCondition() const;
    /// Sets the bottomBoundaryCondition parameter.
    /// The type of boundary conditions to apply on the bottom of the mesh.
    void setBottomBoundaryCondition( const SurfaceFitting3d::BoundaryCondition& bottomBoundaryCondition );

    /// Gets the leftBoundaryCondition parameter.
    /// The type of boundary conditions to apply on the left of the mesh.
    SurfaceFitting3d::BoundaryCondition leftBoundaryCondition() const;
    /// Sets the leftBoundaryCondition parameter.
    /// The type of boundary conditions to apply on the left of the mesh.
    void setLeftBoundaryCondition( const SurfaceFitting3d::BoundaryCondition& leftBoundaryCondition );

    /// Gets the rightBoundaryCondition parameter.
    /// The type of boundary conditions to apply on the right of the mesh.
    SurfaceFitting3d::BoundaryCondition rightBoundaryCondition() const;
    /// Sets the rightBoundaryCondition parameter.
    /// The type of boundary conditions to apply on the right of the mesh.
    void setRightBoundaryCondition( const SurfaceFitting3d::BoundaryCondition& rightBoundaryCondition );

    /// Gets the regularizationType parameter.
    /// The type of regularization used to smooth the mesh.
    SurfaceFitting3d::RegularizationType regularizationType() const;
    /// Sets the regularizationType parameter.
    /// The type of regularization used to smooth the mesh.
    void setRegularizationType( const SurfaceFitting3d::RegularizationType& regularizationType );

    /// Gets the dirichletValues parameter.
    /// The Dirichlet values for the left, right, top, and bottom boundary conditions. If set to 0, the vertices are not moved on the borders for which the Dirichlet conditions are enabled. This value is ignored when the Neumann boundary conditions are enabled for the corresponding mesh border.
    iolink::Vector4d dirichletValues() const;
    /// Sets the dirichletValues parameter.
    /// The Dirichlet values for the left, right, top, and bottom boundary conditions. If set to 0, the vertices are not moved on the borders for which the Dirichlet conditions are enabled. This value is ignored when the Neumann boundary conditions are enabled for the corresponding mesh border.
    void setDirichletValues( const iolink::Vector4d& dirichletValues );

    /// Gets the outputSurface parameter.
    /// The output surface fitting the input volume weights. This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
    std::shared_ptr<iolink::ArrayXd> outputSurface() const;
    /// Sets the outputSurface parameter.
    /// The output surface fitting the input volume weights. This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
    void setOutputSurface( std::shared_ptr<iolink::ArrayXd> outputSurface );

    // Method to launch the command.
    void execute();

};

/// Adjusts the positions of the vertices of a structured quadrilateral mesh according to weights provided by a volume.
/// @param inputIntensityImage The 3D input image containing the weight values to consider for the fitting.
/// @param inputSurface The input mesh that represents the surface to fit. This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
/// @param directions The search direction for each vertex.
/// @param distance The maximum distance to which the vertices can be moved.
/// @param thresholdRange The range of values that can be considered for the best fit.
/// @param smoothing The strength of the regularization applied on the fitted mesh.
/// @param fittingCriterion The criterion according to which a vertex can be kept as the best fit.
/// @param topBoundaryCondition The type of boundary conditions to apply on the top of the mesh.
/// @param bottomBoundaryCondition The type of boundary conditions to apply on the bottom of the mesh.
/// @param leftBoundaryCondition The type of boundary conditions to apply on the left of the mesh.
/// @param rightBoundaryCondition The type of boundary conditions to apply on the right of the mesh.
/// @param regularizationType The type of regularization used to smooth the mesh.
/// @param dirichletValues The Dirichlet values for the left, right, top, and bottom boundary conditions. If set to 0, the vertices are not moved on the borders for which the Dirichlet conditions are enabled. This value is ignored when the Neumann boundary conditions are enabled for the corresponding mesh border.
/// @param outputSurface The output surface fitting the input volume weights. This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
/// @return Returns the outputSurface output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr<iolink::ArrayXd>
surfaceFitting3d( std::shared_ptr< iolink::ImageView > inputIntensityImage,
                  std::shared_ptr<iolink::ArrayXd> inputSurface,
                  std::shared_ptr<iolink::ArrayXd> directions,
                  double distance,
                  const iolink::Vector2d& thresholdRange,
                  double smoothing,
                  SurfaceFitting3d::FittingCriterion fittingCriterion,
                  SurfaceFitting3d::BoundaryCondition topBoundaryCondition,
                  SurfaceFitting3d::BoundaryCondition bottomBoundaryCondition,
                  SurfaceFitting3d::BoundaryCondition leftBoundaryCondition,
                  SurfaceFitting3d::BoundaryCondition rightBoundaryCondition,
                  SurfaceFitting3d::RegularizationType regularizationType,
                  const iolink::Vector4d& dirichletValues,
                  std::shared_ptr<iolink::ArrayXd> outputSurface = nullptr );
} // namespace imagedev
